/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static org.mule.functional.junit4.matchers.ThrowableMessageMatcher.hasMessage;
import static org.mule.service.http.test.netty.AllureConstants.HttpStory.MAX_CONNECTIONS;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.instanceOf;
import static org.junit.Assert.assertThrows;
import static org.junit.internal.matchers.ThrowableCauseMatcher.hasCause;
import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import io.qameta.allure.Story;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import java.io.IOException;
import java.util.concurrent.ExecutionException;

@Story(MAX_CONNECTIONS)
public class HttpClientMaxConnectionsTestCase extends AbstractHttpClientTestCase {

  private static final String ERROR_MESSAGE = "Connection limit exceeded, cannot process request";

  private Latch handling = new Latch();
  private Latch complete = new Latch();
  private HttpClient client;

  public HttpClientMaxConnectionsTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Before
  public void createClient() throws CreateException {
    client = service.getClientFactory().create(new HttpClientConfiguration.Builder()
        .setMaxConnections(1)
        .setName("max-connections-test")
        .build());
    client.start();
  }

  @After
  public void stopClient() {
    if (client != null) {
      client.stop();
    }
  }

  @Override
  protected HttpResponse setUpHttpResponse(HttpRequest request) {
    handling.release();
    try {
      complete.await();
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
    }
    return HttpResponse.builder().build();
  }

  @Test
  public void failsOnConnectionsExceededWhenSync() throws Exception {
    try {
      client.sendAsync(getRequest());
      handling.await();
      var error = assertThrows(IOException.class,
                               () -> client.send(getRequest(), HttpRequestOptions.builder().responseTimeout(1).build()));
      assertThat(error, hasCause(hasMessage(containsString(ERROR_MESSAGE))));
    } finally {
      complete.release();
    }
  }

  @Test
  public void failsOnConnectionsExceededWhenAsync() throws Exception {
    try {
      client.sendAsync(getRequest());
      handling.await();
      var future = client.sendAsync(getRequest(), HttpRequestOptions.builder().responseTimeout(1).build());
      var error = assertThrows(ExecutionException.class, future::get);
      assertThat(error, hasCause(instanceOf(IOException.class)));
      assertThat(error, hasCause(hasMessage(containsString(ERROR_MESSAGE))));

    } finally {
      complete.release();
    }
  }

  private HttpRequest getRequest() {
    return HttpRequest.builder().uri(getUri()).build();
  }

}
