/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static org.mule.runtime.core.api.config.MuleProperties.APP_NAME_PROPERTY;
import static org.mule.runtime.core.api.config.MuleProperties.DOMAIN_NAME_PROPERTY;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.APP;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.DOMAIN;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.PLUGIN;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.POLICY;
import static org.mule.service.http.test.netty.AllureConstants.RESOURCES_ISOLATION;
import static org.mule.service.http.test.netty.AllureConstants.ResourcesIsolationStory.SERVER_ISOLATION;

import static java.lang.Long.valueOf;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThrows;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.MuleConfiguration;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.HttpServerFactory;
import org.mule.runtime.http.api.server.ServerNotFoundException;
import org.mule.service.http.netty.impl.service.ContextHttpServerFactory;
import org.mule.service.http.netty.impl.service.NettyHttpServiceImplementation;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;

import java.util.Optional;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.junit.Test;

@Feature(RESOURCES_ISOLATION)
@Story(SERVER_ISOLATION)
@Issue("W-19160161")
public class ServerFactoryIsolationTestCase extends AbstractHttpServiceTestCase {

  private static final HttpServerConfiguration SERVER_CONFIGURATION = new HttpServerConfiguration.Builder()
      .setName("CONFIG_NAME")
      .setHost("localhost")
      .setPort(8081)
      .build();

  private static final String MULE_CONTEXT_ID = "muleContextId";

  public ServerFactoryIsolationTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Test
  public void unknownArtifactTypeFallbackToOldBehaviour() throws Exception {
    HttpServerFactory httpServerFactory = newServerFactory(empty(), empty(), PLUGIN);
    assertServerFactory(httpServerFactory, MULE_CONTEXT_ID, empty());
    assertThat(httpServerFactory.create(SERVER_CONFIGURATION), is(notNullValue()));
  }

  @Test
  public void appWithNoNameFallbackToOldBehaviour() throws Exception {
    HttpServerFactory httpServerFactory = newServerFactory(empty(), empty(), APP);
    assertServerFactory(httpServerFactory, MULE_CONTEXT_ID, empty());
    assertThat(httpServerFactory.create(SERVER_CONFIGURATION), is(notNullValue()));
  }

  @Test
  public void domainWithNoNameFallbackToOldBehaviour() throws Exception {
    HttpServerFactory httpServerFactory = newServerFactory(empty(), empty(), DOMAIN);
    assertServerFactory(httpServerFactory, MULE_CONTEXT_ID, empty());
    assertThat(httpServerFactory.create(SERVER_CONFIGURATION), is(notNullValue()));
  }

  @Test
  public void policyWithNoNameFallbackToOldBehaviour() throws Exception {
    HttpServerFactory httpServerFactory = newServerFactory(empty(), empty(), POLICY);
    assertServerFactory(httpServerFactory, MULE_CONTEXT_ID, empty());
    assertThat(httpServerFactory.create(SERVER_CONFIGURATION), is(notNullValue()));
  }

  @Test
  public void sameAppAccessSameServer() throws Exception {
    final String app = "app";
    HttpServerFactory factory = newAppServerFactory(app, empty());

    // No failure means OK. Can't compare servers because they are different implementations
    assertThat(factory.create(SERVER_CONFIGURATION), is(notNullValue()));
    assertThat(factory.lookup(SERVER_CONFIGURATION.getName()), is(notNullValue()));
  }

  @Test
  public void sameDomainAccessSameServer() throws Exception {
    final String domain = "domain";
    HttpServerFactory factory = newDomainServerFactory(domain);

    // No failure means OK. Can't compare servers because they are different implementations
    assertThat(factory.create(SERVER_CONFIGURATION), is(notNullValue()));
    assertThat(factory.lookup(SERVER_CONFIGURATION.getName()), is(notNullValue()));
  }

  @Test
  public void samePolicyAccessSameServer() throws Exception {
    final String policy = "policy";
    HttpServerFactory factory = newPolicyServerFactory(policy);

    // No failure means OK. Can't compare servers because they are different implementations
    assertThat(factory.create(SERVER_CONFIGURATION), is(notNullValue()));
    assertThat(factory.lookup(SERVER_CONFIGURATION.getName()), is(notNullValue()));
  }

  @Test
  public void appsAccessServerFromDomain() throws Exception {
    final String app1 = "app1";
    final String app2 = "app2";
    final String domain = "domain";

    HttpServerFactory appFactory1 = newAppServerFactory(app1, of(domain));
    HttpServerFactory appFactory2 = newAppServerFactory(app2, of(domain));
    HttpServerFactory domainFactory = newDomainServerFactory(domain);

    // No failure means OK. Can't compare servers because they are different implementations
    assertThat(domainFactory.create(SERVER_CONFIGURATION), is(notNullValue()));
    assertThat(appFactory1.lookup(SERVER_CONFIGURATION.getName()), is(notNullValue()));
    assertThat(appFactory2.lookup(SERVER_CONFIGURATION.getName()), is(notNullValue()));
  }

  @Test
  public void policyCantAccessAppServer() throws Exception {
    final String policy = "policy";
    final String app = "app";

    HttpServerFactory appFactory = newAppServerFactory(app, empty());
    HttpServerFactory policyFactory = newPolicyServerFactory(policy);

    assertThat(appFactory.create(SERVER_CONFIGURATION), is(notNullValue()));

    assertThrows(ServerNotFoundException.class, () -> policyFactory.lookup(SERVER_CONFIGURATION.getName()));
  }

  @Test
  public void policyCantAccessAppServerWithSameNames() throws Exception {
    final String name = "sameName";

    HttpServerFactory appFactory = newAppServerFactory(name, empty());
    HttpServerFactory policyFactory = newPolicyServerFactory(name);

    assertThat(appFactory.create(SERVER_CONFIGURATION), is(notNullValue()));

    assertThrows(ServerNotFoundException.class, () -> policyFactory.lookup(SERVER_CONFIGURATION.getName()));
  }

  @Test
  public void appCantAccessDomainServerWithSameNames() throws Exception {
    final String name = "sameName";

    HttpServerFactory appFactory = newAppServerFactory(name, empty());
    HttpServerFactory domainFactory = newDomainServerFactory(name);

    assertThat(domainFactory.create(SERVER_CONFIGURATION), is(notNullValue()));

    assertThrows(ServerNotFoundException.class, () -> appFactory.lookup(SERVER_CONFIGURATION.getName()));
  }

  @Test
  public void differentAppsDontAccessSameServers() throws Exception {
    final String app1 = "app1";
    final String app2 = "app2";
    HttpServerFactory factory1 = newAppServerFactory(app1, empty());
    HttpServerFactory factory2 = newAppServerFactory(app2, empty());

    assertThat(factory1.create(SERVER_CONFIGURATION), is(notNullValue()));
    assertThrows(ServerNotFoundException.class, () -> factory2.lookup(SERVER_CONFIGURATION.getName()));
  }

  private HttpServerFactory newAppServerFactory(String appName, Optional<String> domainName) {
    return newServerFactory(of(appName), domainName, APP);
  }

  private HttpServerFactory newDomainServerFactory(String domainName) {
    return newServerFactory(empty(), of(domainName), DOMAIN);
  }

  private HttpServerFactory newPolicyServerFactory(String policyName) {
    return newServerFactory(of(policyName), empty(), POLICY);
  }

  protected HttpServerFactory newServerFactory(Optional<String> artifactName, Optional<String> domainName,
                                               ArtifactType artifactType) {
    MuleConfiguration muleConfiguration = mock(MuleConfiguration.class);
    when(muleConfiguration.getShutdownTimeout()).thenReturn(valueOf(20));

    MuleContext muleContext = mock(MuleContext.class);
    when(muleContext.getArtifactType()).thenReturn(artifactType);
    when(muleContext.getId()).thenReturn(MULE_CONTEXT_ID);
    when(muleContext.getConfiguration()).thenReturn(muleConfiguration);

    Registry registry = mock(Registry.class);
    when(registry.<String>lookupByName(APP_NAME_PROPERTY)).thenReturn(artifactName);
    when(registry.<String>lookupByName(DOMAIN_NAME_PROPERTY)).thenReturn(domainName);
    return ((NettyHttpServiceImplementation) service).getServerFactory(muleContext, registry);
  }

  protected void assertServerFactory(HttpServerFactory serverFactory, String contextName, Optional<String> parentContext) {
    assertThat(((ContextHttpServerFactory) serverFactory).getContextName(), is(contextName));
    assertThat(((ContextHttpServerFactory) serverFactory).getParentContextName(), is(parentContext.orElse(null)));
  }
}
