/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.mule.service.http.test.util.ClassUtils.instantiateClass;

import static junit.framework.TestCase.fail;

import static java.util.Collections.singletonList;

import static com.github.peterwippermann.junit4.parameterizedsuite.ParameterContext.getParameter;
import static com.github.peterwippermann.junit4.parameterizedsuite.ParameterContext.isParameterSet;

import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.service.http.netty.impl.service.NettyHttpServiceImplementation;
import org.mule.tck.SimpleUnitTestSupportSchedulerService;
import org.mule.tck.junit4.AbstractMuleTestCase;

import org.junit.After;
import org.junit.Before;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

/**
 * Base class for all HTTP service functional tests. The service implementation will be loaded based on it's class name, allowing
 * tests to customize it.
 */
@RunWith(Parameterized.class)
public abstract class AbstractHttpServiceTestCase extends AbstractMuleTestCase {

  public String serviceToLoad;

  protected HttpService service;
  private SchedulerService schedulerService;

  @Parameters(name = "{0}")
  public static Iterable<Object[]> params() {
    if (isParameterSet()) {
      return singletonList(getParameter(Object[].class));
    } else {
      return singletonList(new String[] {NettyHttpServiceImplementation.class.getName()});
    }
  }

  public AbstractHttpServiceTestCase(String serviceToLoad) {
    this.serviceToLoad = serviceToLoad;
  }

  @Before
  public void createServices() throws Exception {
    schedulerService = getSchedulerService();
    service = (HttpService) instantiateClass(serviceToLoad, new Object[] {schedulerService},
                                             this.getClass().getClassLoader());
    startIfNeeded(service);
  }

  protected SchedulerService getSchedulerService() {
    return new SimpleUnitTestSupportSchedulerService();
  }

  @After
  public void closeServices() throws Exception {
    if (service != null) {
      stopIfNeeded(service);
    }
    stopIfNeeded(schedulerService);
  }

  public static class IgnoreResponseStatusCallback implements ResponseStatusCallback {

    @Override
    public void responseSendFailure(Throwable throwable) {
      fail(throwable.getMessage());
    }

    @Override
    public void responseSendSuccessfully() {
      // Do nothing
    }
  }
}
