/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server;

import static org.mule.runtime.api.metadata.MediaType.TEXT;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.fail;
import static java.lang.String.format;
import static java.lang.Thread.sleep;
import static java.util.Collections.singletonList;
import static java.util.concurrent.Executors.newCachedThreadPool;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.tck.junit4.rule.SystemProperty;
import org.apache.http.StatusLine;
import org.apache.http.client.fluent.Request;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

@Ignore("W-19471841")
public class SlowResponsesHandlingTestCase extends AbstractHttpServerTestCase {

  @Rule
  public SystemProperty selectorsTimeout = new SystemProperty("mule.timeoutToUseSelectorWhileStreamingResponseMillis", "0");

  private static final long TIME_PER_BYTE = 100;
  private static final int REQUESTS = 100;
  private static final int LATCH_TIMEOUT = 5000;
  private static final String ENDPOINT = "/firstSlow";
  private AtomicInteger requestNumber = new AtomicInteger(0);
  private static CountDownLatch start = new CountDownLatch(1);
  private final CountDownLatch latch = new CountDownLatch(REQUESTS);
  private final ExecutorService executor = newCachedThreadPool();

  public SlowResponsesHandlingTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected void setUpServer() throws Exception {
    server = service.getServerFactory().create(configureServer(new HttpServerConfiguration.Builder()
        .setHost("localhost")
        .setPort(port.getNumber())
        .setName(getServerName())
        .setSchedulerSupplier(() -> getSchedulerService().ioScheduler()))
            .build());
    server.start();
  }

  @Before
  public void setUp() throws Exception {
    setUpServer();
    registerHandler("Success!");
  }

  private void registerHandler(String payload) {
    server.addRequestHandler(singletonList(GET.name()), ENDPOINT, (requestContext, responseCallback) -> {
      InputStream response;
      if (requestNumber.getAndIncrement() == 0) {
        start.countDown();
        response = new InputStream() {

          private int numreads = 0;

          @Override
          public int read() throws IOException {
            try {
              sleep(TIME_PER_BYTE);
              return numreads++ <= (LATCH_TIMEOUT / TIME_PER_BYTE + 10) ? 1 : -1;
            } catch (InterruptedException e) {
              return -1;
            }
          }
        };

      } else {
        response = new ByteArrayInputStream(payload.getBytes());
      }
      responseCallback
          .responseReady(HttpResponse.builder().entity(new InputStreamHttpEntity(response))
              .addHeader(CONTENT_TYPE, TEXT.toRfcString())
              .build(), new IgnoreResponseStatusCallback());
    }).start();
  }

  @Override
  protected String getServerName() {
    return "slow-responses";
  }

  @Test
  public void onlySlowRequestFails() throws Exception {
    AtomicInteger correctCount = new AtomicInteger(0);
    performRequestInThread(null);
    start.await();
    // These should be all correct
    for (int i = 0; i < REQUESTS; i++) {
      performRequestInThread(correctCount);
    }

    if (!latch.await(LATCH_TIMEOUT, TimeUnit.MILLISECONDS)) {
      fail(format("We only got %d responses out of %d", correctCount.get(), REQUESTS));
    }
  }

  protected String getUrl(String path) {
    return format("http://%s:%s/%s", server.getServerAddress().getIp(), port.getNumber(), path);
  }

  private void performRequestInThread(AtomicInteger counter) {
    executor.submit(() -> {
      try {
        Request request = Request.Get(getUrl(ENDPOINT));
        org.apache.http.HttpResponse response = request.execute().returnResponse();
        StatusLine statusLine = response.getStatusLine();
        assertThat(statusLine.getStatusCode(), is(OK.getStatusCode()));
        if (counter != null) {
          counter.incrementAndGet();
          latch.countDown();
        }
      } catch (Exception e) {
        //
      }
    });
  }

}
