/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.message;

import static org.mule.service.http.netty.impl.client.auth.ntlm.NtlmFlags.NTLMSSP_NEGOTIATE_UNICODE;

import java.io.IOException;

/**
 * Represents a Type 2 NTLM message used in NTLM authentication.
 * <p>
 * This message is sent by the server to the client as a response to a Type 1 message. It contains information such as the
 * server's challenge and the target information needed for further authentication.
 * </p>
 * <p>
 * This implementation is based on the jcifs library, available at <a href="https://github.com/kevinboone/jcifs">jcifs GitHub
 * Repository</a>.
 * </p>
 */
public class Type2Message extends NtlmMessage {

  private byte[] targetInformation;
  private byte[] challenge;
  private byte[] context;
  private String target;

  /**
   * Creates a Type-2 message using the given raw Type-2 material.
   *
   * @param material The raw Type-2 material used to construct this message.
   * @throws IOException If an error occurs while parsing the material.
   */
  public Type2Message(byte[] material) throws IOException {
    parse(material);
  }

  private void parse(byte[] input) throws IOException {
    int pos = 0;
    for (int i = 0; i < 8; i++) {
      if (input[i] != NTLMSSP_SIGNATURE[i]) {
        throw new IOException("Not an NTLMSSP message.");
      }
    }
    pos += 8;

    if (readULong(input, pos) != NTLMSSP_TYPE2) {
      throw new IOException("Not a Type 2 message.");
    }
    pos += 4;

    int flags = readULong(input, pos + 8);
    setFlags(flags);

    byte[] targetName = readSecurityBuffer(input, pos);
    int targetNameOff = readULong(input, pos + 4);
    if (targetName.length != 0) {
      setTarget(new String(targetName, ((flags & NTLMSSP_NEGOTIATE_UNICODE) != 0) ? UNI_ENCODING : getOEMEncoding()));
    }
    pos += 12; // 8 for target, 4 for flags

    if (!allZeros8(input, pos)) {
      byte[] challengeBytes = new byte[8];
      System.arraycopy(input, pos, challengeBytes, 0, challengeBytes.length);
      setChallenge(challengeBytes);
    }
    pos += 8;

    if (targetNameOff < pos + 8 || input.length < pos + 8) {
      // no room for Context/Reserved
      return;
    }

    if (!allZeros8(input, pos)) {
      byte[] contextBytes = new byte[8];
      System.arraycopy(input, pos, contextBytes, 0, contextBytes.length);
      setContext(contextBytes);
    }
    pos += 8;

    if (targetNameOff < pos + 8 || input.length < pos + 8) {
      // no room for target info
      return;
    }

    byte[] targetInfo = readSecurityBuffer(input, pos);
    if (targetInfo.length != 0) {
      setTargetInformation(targetInfo);
    }
  }

  /**
   * Sets the challenge for this message.
   *
   * @param challenge The challenge from the domain controller/server.
   */
  public void setChallenge(byte[] challenge) {
    this.challenge = challenge;
  }

  public void setTarget(String target) {
    this.target = target;
  }

  /**
   * Sets the target information block. The target information block is used by the client to create an NTLMv2 response.
   *
   * @param targetInformation The target information block.
   */
  public void setTargetInformation(byte[] targetInformation) {
    this.targetInformation = targetInformation;
  }

  private static boolean allZeros8(byte[] input, int pos) {
    for (int i = pos; i < pos + 8; i++) {
      if (input[i] != 0) {
        return false;
      }
    }
    return true;
  }

  public void setContext(byte[] context) {
    this.context = context;
  }

  public byte[] getChallenge() {
    return challenge;
  }

  public byte[] getTargetInformation() {
    return targetInformation;
  }

  public byte[] getContext() {
    return context;
  }

  public String getTarget() {
    return target;
  }
}
