/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.utils;

import static io.netty.handler.codec.http.HttpResponseStatus.FOUND;
import static io.netty.handler.codec.http.HttpResponseStatus.valueOf;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.service.http.netty.impl.util.RedirectHelper;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

import io.netty.handler.codec.http.DefaultHttpHeaders;
import io.netty.handler.codec.http.HttpHeaders;
import io.qameta.allure.Issue;
import org.junit.Before;
import org.junit.Test;
import reactor.netty.http.client.HttpClientRequest;
import reactor.netty.http.client.HttpClientResponse;

public class RedirectHelperTestCase extends AbstractMuleTestCase {

  private HttpClientRequest mockRequest;
  private HttpClientResponse mockResponse;

  @Before
  public void setUp() {
    mockRequest = mock(HttpClientRequest.class);
    mockResponse = mock(HttpClientResponse.class);
  }

  private Map<String, String> getCookiesMap(RedirectHelper redirectHelper) throws Exception {
    Field field = redirectHelper.getClass().getDeclaredField("cookiesMap");
    field.setAccessible(true);
    return (Map<String, String>) field.get(redirectHelper);
  }

  @Test
  @Issue("W-17405565")
  public void testHandleRedirectResponseWhenCookiesAreNotExpired() throws Exception {
    HttpHeaders headers = new DefaultHttpHeaders();
    headers.add("Set-Cookie", "session=abcd1234; Path=/; Expires=Sun, 09 Jun 2030 10:18:14 GMT");
    headers.add("Set-Cookie", "user=testUser; Path=/; Max-Age=3600");

    when(mockResponse.responseHeaders()).thenReturn(headers);
    when(mockResponse.status()).thenReturn(FOUND);
    RedirectHelper redirectHelper = new RedirectHelper(new DefaultHttpHeaders());

    redirectHelper.handleRedirectResponse(mockResponse);

    Map<String, String> cookiesMap = getCookiesMap(redirectHelper);

    assertThat(cookiesMap.size(), is(2));
    assertThat(cookiesMap.containsKey("session"), is(true));
    assertThat(cookiesMap.get("session"), is("abcd1234"));

    assertThat(cookiesMap.containsKey("user"), is(true));
    assertThat(cookiesMap.get("user"), is("testUser"));
  }

  @Test
  @Issue("W-17405565")
  public void testHandleRedirectResponseWhenCookiesAreExpired() throws Exception {
    HttpHeaders headers = new DefaultHttpHeaders();
    headers.add("Set-Cookie", "session=abcd1234; Path=/; Expires=Sun, 09 Jun 2024 10:18:14 GMT");
    headers.add("Set-Cookie", "user=testUser; Path=/; Max-Age=0");

    when(mockResponse.responseHeaders()).thenReturn(headers);
    when(mockResponse.status()).thenReturn(FOUND);
    RedirectHelper redirectHelper = new RedirectHelper(new DefaultHttpHeaders());

    redirectHelper.handleRedirectResponse(mockResponse);

    Map<String, String> cookiesMap = getCookiesMap(redirectHelper);

    assertThat(cookiesMap.size(), is(0));
  }

  @Test
  @Issue("W-17908187")
  public void testHandleRedirectResponse_WhenInitialCookieArePresent() throws Exception {
    Map.Entry<String, String> cookie = new HashMap.SimpleEntry<>("user", "testUser");

    HttpHeaders headers = new DefaultHttpHeaders();
    headers.add("Set-Cookie", "session=abcd1234; Path=/; Expires=Sun, 09 Jun 2030 10:18:14 GMT");

    when(mockResponse.responseHeaders()).thenReturn(headers);
    when(mockResponse.status()).thenReturn(FOUND);
    RedirectHelper redirectHelper =
        new RedirectHelper(new DefaultHttpHeaders().add("Cookie", cookie.getKey() + "=" + cookie.getValue()));

    redirectHelper.handleRedirectResponse(mockResponse);

    Map<String, String> cookiesMap = getCookiesMap(redirectHelper);

    assertThat(cookiesMap.size(), is(2));
    assertThat(cookiesMap.containsKey("session"), is(true));
    assertThat(cookiesMap.get("session"), is("abcd1234"));

    assertThat(cookiesMap.containsKey("user"), is(true));
    assertThat(cookiesMap.get("user"), is("testUser"));
  }

  @Test
  @Issue("W-17405565")
  public void testAddCookie_ToRedirectedRequest_NoCookiesPresent() throws Exception {
    when(mockRequest.requestHeaders()).thenReturn(new DefaultHttpHeaders());
    RedirectHelper redirectHelper = new RedirectHelper(new DefaultHttpHeaders());

    Map<String, String> cookiesMap = getCookiesMap(redirectHelper);
    assertThat(cookiesMap.isEmpty(), is(true));

    redirectHelper.addCookiesToRedirectedRequest(mockRequest);

    verify(mockRequest, never()).addHeader(anyString(), anyString());
  }

  @Test
  @Issue("W-17898850")
  public void shouldNotChangeMethodOnNewHelper() {
    RedirectHelper redirectHelper = new RedirectHelper(new DefaultHttpHeaders());
    assertThat(redirectHelper.shouldChangeMethod(), is(false));
  }

  @Test
  @Issue("W-17898850")
  public void shouldChangeMethodForEachStatus() {
    RedirectHelper redirectHelper = new RedirectHelper(new DefaultHttpHeaders());

    when(mockResponse.responseHeaders()).thenReturn(new DefaultHttpHeaders());

    for (int statusCode = 300; statusCode <= 308; ++statusCode) {
      when(mockResponse.status()).thenReturn(valueOf(statusCode));
      redirectHelper.handleRedirectResponse(mockResponse);
      boolean expected = (statusCode == 302 || statusCode == 303);
      assertThat(redirectHelper.shouldChangeMethod(), is(expected));
    }
  }
}
