/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalToCompressingWhiteSpace;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.test.netty.tck.ExecutorRule;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.TcpTextClient;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.net.InetSocketAddress;

import io.qameta.allure.Issue;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

@Issue("W-17413024")
public class ServerReadTimeoutTestCase extends AbstractMuleTestCase {

  @ClassRule
  public static ExecutorRule executorRule = new ExecutorRule();

  private static final long SMALL_READ_TIMEOUT_MILLIS = 100L;

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  private HttpServer httpServer;

  @Before
  public void setup() throws Exception {
    HttpListenerRegistry listenerRegistry = new HttpListenerRegistry();
    httpServer = NettyHttpServer.builder()
        .withName("test-server")
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withHttpListenerRegistry(listenerRegistry)
        .withShutdownTimeout(() -> 5000L)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, "test-server", true, 30000,
                                                                           SMALL_READ_TIMEOUT_MILLIS, null, 300,
                                                                           executorRule.getExecutor()))
        .build();
    httpServer.start();
    httpServer.addRequestHandler("/test", (requestContext, responseCallback) -> {
      executorRule.getExecutor().submit(() -> {
        String asString = IOUtils.toString(requestContext.getRequest().getEntity().getContent());
        responseCallback.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity(asString)),
                                       new NoOpResponseStatusCallback());
      });
    });
  }

  @After
  public void tearDown() {
    if (!httpServer.isStopped()) {
      httpServer.stop();
    }
    httpServer.dispose();
  }

  @Test
  public void sendPartialRequestShouldTimeout() throws IOException {
    try (TcpTextClient tcpTextClient = new TcpTextClient("localhost", serverPort.getNumber())) {
      // Send only one chunk...
      tcpTextClient.sendString("""
          POST /test HTTP/1.1\r
          Host: localhost: %d\r
          Transfer-Encoding: chunked\r
          \r
          7\r
          Partial\r
          """.formatted(serverPort.getNumber()));

      // The response should be received normally.
      String header = tcpTextClient.receiveUntil("\r\n\r\n");
      assertThat(header, equalToCompressingWhiteSpace("""
          HTTP/1.1 408 Request Timeout\r
          Connection: close\r
          Content-Length: 23"""));

      String content = tcpTextClient.receiveUntil("\r\n\r\n");
      assertThat(content, is("Timeout reading request"));

      String nothing = tcpTextClient.receiveUntil("\r\n\r\n");
      assertThat(nothing, is(""));
    }
  }
}
