/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static java.util.Collections.singletonList;

import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.channels.UnsupportedAddressTypeException;
import java.util.List;

import io.netty.resolver.AddressResolver;
import io.netty.resolver.AddressResolverGroup;
import io.netty.util.concurrent.EventExecutor;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.Promise;

/**
 * A custom {@link AddressResolverGroup} implementation that acts as a no-operation (no-op) resolver for
 * {@link InetSocketAddress}.
 *
 * This resolver is used in scenarios where a proxy is configured, and DNS resolution should be bypassed on the client side. By
 * directly returning the provided {@link InetSocketAddress}, it avoids performing any DNS lookups, delegating the responsibility
 * of DNS resolution to the proxy server.
 */
public final class InetNoopAddressResolverGroup extends AddressResolverGroup<InetSocketAddress> {

  public static final InetNoopAddressResolverGroup INSTANCE = new InetNoopAddressResolverGroup();

  private InetNoopAddressResolverGroup() {}

  @Override
  protected AddressResolver<InetSocketAddress> newResolver(EventExecutor executor) throws Exception {
    return new AddressResolver<InetSocketAddress>() {

      @Override
      public boolean isSupported(SocketAddress address) {
        return address instanceof InetSocketAddress;
      }

      @Override
      public boolean isResolved(SocketAddress address) {
        return address instanceof InetSocketAddress;
      }

      @Override
      public Future<InetSocketAddress> resolve(SocketAddress address) {
        if (address instanceof InetSocketAddress) {
          return executor.newSucceededFuture((InetSocketAddress) address);
        } else {
          return executor.newFailedFuture(new UnsupportedAddressTypeException());
        }
      }

      @Override
      public Future<InetSocketAddress> resolve(SocketAddress address, Promise<InetSocketAddress> promise) {
        if (address instanceof InetSocketAddress) {
          promise.setSuccess((InetSocketAddress) address);
        } else {
          promise.setFailure(new UnsupportedAddressTypeException());
        }
        return promise;
      }

      @Override
      public Future<List<InetSocketAddress>> resolveAll(SocketAddress address) {
        if (address instanceof InetSocketAddress) {
          return executor.newSucceededFuture(singletonList((InetSocketAddress) address));
        } else {
          return executor.newFailedFuture(new UnsupportedAddressTypeException());
        }
      }

      @Override
      public Future<List<InetSocketAddress>> resolveAll(SocketAddress address, Promise<List<InetSocketAddress>> promise) {
        if (address instanceof InetSocketAddress) {
          promise.setSuccess(singletonList((InetSocketAddress) address));
        } else {
          promise.setFailure(new UnsupportedAddressTypeException());
        }
        return promise;
      }

      @Override
      public void close() {}
    };
  }
}
