/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;


import static org.mule.service.http.netty.impl.client.ReactorNettyClient.ALWAYS_SEND_BODY_KEY;
import static org.mule.service.http.netty.impl.client.ReactorNettyClient.REDIRECT_CHANGE_METHOD;

import static java.lang.Boolean.FALSE;
import static java.lang.Boolean.TRUE;

import static com.google.common.net.HttpHeaders.CONTENT_LENGTH;
import static com.google.common.net.HttpHeaders.TRANSFER_ENCODING;

import io.netty.channel.ChannelDuplexHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpRequest;

/**
 * A Netty handler that intercepts HTTP responses and modifies the HTTP method of outgoing requests following a 302 Found or a 303
 * SEE_OTHER redirection. <a href="https://tools.ietf.org/html/rfc7231#section-6.4.3">RFC 7231</a>.
 */
public class RedirectMethodChangeHandler extends ChannelDuplexHandler {

  @Override
  public void write(ChannelHandlerContext ctx, Object msg, ChannelPromise promise) throws Exception {
    if (msg instanceof HttpRequest request && shouldChangeMethodToGET(ctx)) {
      request.setMethod(HttpMethod.GET);
      if (shouldOverrideBodyWithEmpty(ctx)) {
        request.headers().remove(TRANSFER_ENCODING);
        request.headers().set(CONTENT_LENGTH, "0");
      }
    }
    super.write(ctx, msg, promise);
  }

  private static boolean shouldOverrideBodyWithEmpty(ChannelHandlerContext ctx) {
    return FALSE.equals(ctx.channel().attr(ALWAYS_SEND_BODY_KEY).get());
  }

  private static boolean shouldChangeMethodToGET(ChannelHandlerContext ctx) {
    return TRUE.equals(ctx.channel().attr(REDIRECT_CHANGE_METHOD).get());
  }
}

