/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth;

import java.util.Base64;

/**
 * Constructs the basic auth headers depending on the different exchanges between client and server, and if the mode should be
 * preemptive or not
 */
public class BasicAuthHeaderFactory implements AuthHeaderFactory {

  // Status of the authentication, to avoid falling into bad states (as infinite recursions) if the server is not well
  // implemented.
  enum Status {

    // Initial state, the first request wasn't sent yet.
    NOT_STARTED,

    // When not preemptive, we have to send the headers in a subsequent request.
    NO_HEADERS_SENT_BECAUSE_NOT_PREEMPTIVE,

    // We already sent the header, so the next request doesn't need authentication. If it's a 401, it means that auth failed.
    FINISHED,
  }

  private final String username;
  private final String password;
  private final boolean isPreemptive;
  private Status status;

  public BasicAuthHeaderFactory(boolean preemptive, String username, String password) {
    this.isPreemptive = preemptive;
    this.username = username;
    this.password = password;
    this.status = Status.NOT_STARTED;
  }

  private String getBasicAuthHeader() {
    String authString = this.username + ":" + this.password;
    return "Basic " + Base64.getEncoder().encodeToString(authString.getBytes());
  }

  @Override
  public boolean hasFinished() {
    return Status.FINISHED == status;
  }

  @Override
  public String getNextHeader(String wwwAuthenticateHeader) {
    switch (status) {
      case NOT_STARTED:
        if (isPreemptive) {
          status = Status.FINISHED;
          return getBasicAuthHeader();
        } else {
          status = Status.NO_HEADERS_SENT_BECAUSE_NOT_PREEMPTIVE;
          return null;
        }
      case NO_HEADERS_SENT_BECAUSE_NOT_PREEMPTIVE:
        if (isPreemptive) {
          throw new IllegalStateException("BasicAuthHeaderFactory reached an invalid state");
        } else {
          status = Status.FINISHED;
          return getBasicAuthHeader();
        }
      case FINISHED:
        return null;
    }
    return null;
  }
}
