/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.util;

import static javax.crypto.Cipher.ENCRYPT_MODE;

import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.SecretKeySpec;

/**
 * Utility class providing cryptographic functions for NTLM authentication. Includes HMAC-T64, MD5 hashing, and RC4 encryption
 * methods.
 *
 * <p>
 * This implementation is based on the jcifs library, available at:
 * <a href="https://github.com/codelibs/jcifs">https://github.com/codelibs/jcifs</a>
 * </p>
 *
 * @see <a href="https://github.com/codelibs/jcifs">jcifs on GitHub</a>
 */
public class Crypto {

  /** Constant for the RC4 encryption algorithm name. */
  public static final String RC4 = "RC4";

  /** Constant for the MD5 hashing algorithm name. */
  public static final String MD5 = "MD5";

  /**
   * Creates a custom HMAC-T64 MessageDigest with the specified key.
   *
   * @param key the key for HMAC-T64 initialization.
   * @return a {@link MessageDigest} instance initialized for HMAC-T64.
   * @throws NoSuchAlgorithmException if the HMAC-T64 algorithm is unavailable.
   */
  public static MessageDigest getHMACT64(byte[] key) throws NoSuchAlgorithmException {
    return new HMACT64(key);
  }

  /**
   * Returns an MD5 MessageDigest instance for MD5 hashing.
   *
   * @return a {@link MessageDigest} instance configured for MD5.
   * @throws NoSuchAlgorithmException if MD5 is unavailable.
   */
  public static MessageDigest getMD5() throws NoSuchAlgorithmException {
    return MessageDigest.getInstance(MD5);
  }

  /**
   * Initializes an RC4 Cipher instance with the specified key.
   *
   * @param key the encryption key.
   * @return a {@link Cipher} instance configured for RC4 encryption.
   * @throws InvalidKeyException      if the provided key is invalid.
   * @throws NoSuchPaddingException   if padding is unavailable for RC4.
   * @throws NoSuchAlgorithmException if RC4 is unavailable.
   */
  public static Cipher getArcfour(byte[] key) throws InvalidKeyException, NoSuchPaddingException, NoSuchAlgorithmException {
    Cipher c = Cipher.getInstance(RC4);
    c.init(ENCRYPT_MODE, new SecretKeySpec(key, RC4));
    return c;
  }
}
