/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.proxy;

import static org.slf4j.LoggerFactory.getLogger;
import static reactor.netty.NettyPipeline.HttpCodec;

import org.mule.runtime.http.api.client.proxy.ProxyConfig;

import java.net.InetSocketAddress;
import java.net.SocketAddress;

import org.slf4j.Logger;

import io.netty.channel.ChannelPipeline;

/**
 * Configures a {@link ChannelPipeline} with the corresponding (if any) to the given {@link ProxyConfig}. The handler could be:
 * <il>
 * <li>none, if the config is null or the host where the client is connecting is in the list of non-proxy-hosts.</li>
 * <li>a {@link BlindTunnelingProxyClientHandler}, if the host is not a "non-proxy-host" and the user passes
 * useTunneling=true.</li>
 * <li>a {@link MessageForwardingProxyClientHandler}, if the host is not a "non-proxy-host" and the user passes
 * useTunneling=false.</li> </il>
 */
public class ProxyPipelineConfigurer {

  private static final Logger LOGGER = getLogger(ProxyPipelineConfigurer.class);

  private final ProxyConfig proxyConfig;

  public ProxyPipelineConfigurer(ProxyConfig proxyConfig) {
    this.proxyConfig = proxyConfig;
  }

  public void configurePipeline(ChannelPipeline pipeline, SocketAddress remoteAddress, boolean useTunneling) {
    if (proxyConfig == null) {
      // No proxy.
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("No proxy configured for channel {}", pipeline.channel().id().asLongText());
      }
      return;
    }

    if (isNonProxyHost(proxyConfig.getNonProxyHosts(), remoteAddress)) {
      // Proxy configured, but not for this particular host.
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Proxy skipped in channel {} because nonProxyHost matches address {}", pipeline.channel().id().asLongText(),
                     remoteAddress);
      }
      return;
    }

    if (useTunneling) {
      // Establish a blind forwarding proxy tunnel (with CONNECT method).
      // See RFC https://datatracker.ietf.org/doc/html/rfc7231#section-4.3.6
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Establish a blind forwarding proxy tunnel (with CONNECT method) for channel {}",
                     pipeline.channel().id().asLongText());
      }
      pipeline.addFirst("TunnelProxyHandler", new BlindTunnelingProxyClientHandler(proxyConfig));
      return;
    }

    // Otherwise, we use a message-forwarding HTTP proxy agent
    // See RFC https://datatracker.ietf.org/doc/html/rfc7230#page-10
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Establish a message-forwarding proxy (without CONNECT method) for channel {}",
                   pipeline.channel().id().asLongText());
    }
    pipeline.addAfter(HttpCodec, "ProxyAuthHandler", new MessageForwardingProxyClientHandler(proxyConfig));
  }

  private boolean isNonProxyHost(String nonProxyHosts, SocketAddress socketAddress) {
    if (nonProxyHosts == null || nonProxyHosts.isEmpty()) {
      return false;
    }

    if (!(socketAddress instanceof InetSocketAddress)) {
      // this should never happen, but if we don't know that it's a non-proxy-host, then we return false.
      return false;
    }

    final String hostString = ((InetSocketAddress) socketAddress).getHostString();
    for (String nonProxyHost : nonProxyHosts.split(",")) {
      if (nonProxyHost.trim().equals(hostString)) {
        return true;
      }
    }

    return false;
  }
}
