/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import java.util.function.BiConsumer;
import java.util.function.Consumer;

import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPipeline;
import io.netty.handler.codec.http2.Http2StreamChannel;
import io.netty.handler.ssl.ApplicationProtocolNames;
import io.netty.handler.ssl.ApplicationProtocolNegotiationHandler;
import io.netty.handler.ssl.SslHandler;

/**
 * Rejects the requests if HTTP/1 isn't going to be used. Once filtered, a multiplexer is added to the pipeline so that new
 * {@link Http2StreamChannel} instances are initialized by a {@link MultiplexerChannelInitializer}.
 *
 * @see MultiplexerChannelInitializer for more information.
 */
public class HttpWithAPNServerHandler extends ApplicationProtocolNegotiationHandler {

  private final BiConsumer<ChannelPipeline, SslHandler> http1Configurer;
  private final SslHandler sslHandler;

  public HttpWithAPNServerHandler(BiConsumer<ChannelPipeline, SslHandler> http1Configurer, SslHandler sslHandler) {
    super(ApplicationProtocolNames.HTTP_1_1);
    this.http1Configurer = http1Configurer;
    this.sslHandler = sslHandler;
  }

  @Override
  protected void configurePipeline(ChannelHandlerContext ctx, String protocol) throws Exception {
    if (ApplicationProtocolNames.HTTP_1_1.equals(protocol)) {
      http1Configurer.accept(ctx.pipeline(), sslHandler);
      return;
    }

    throw new IllegalStateException("Unknown protocol: " + protocol);
  }
}
