/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server.util;

import static org.mule.runtime.api.metadata.MediaType.MULTIPART_FORM_DATA;
import static org.mule.runtime.api.metadata.MediaType.MULTIPART_RELATED;
import static org.mule.runtime.core.api.util.StringUtils.WHITE_SPACE;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_DISPOSITION;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_ID;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Values.BOUNDARY;

import static java.lang.String.format;
import static java.net.URLDecoder.decode;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.singletonMap;
import static java.util.UUID.randomUUID;
import static java.util.regex.Pattern.compile;

import static org.apache.commons.io.IOUtils.toByteArray;

import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;
import org.mule.runtime.http.api.domain.entity.multipart.MultipartHttpEntity;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Consumer;
import java.util.regex.Pattern;

import javax.activation.DataHandler;
import javax.mail.BodyPart;
import javax.mail.Header;
import javax.mail.MessagingException;
import javax.mail.MultipartDataSource;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import javax.mail.util.ByteArrayDataSource;

import com.google.common.collect.Lists;

import org.apache.commons.io.IOUtils;

public class HttpParser {

  private static final Pattern SPACE_ENTITY_OR_PLUS_SIGN_REGEX = compile("%20|\\+");
  private static final String NAME_ATTRIBUTE = "name";

  public static String extractPath(String uri) {
    String path = uri;
    int i = path.indexOf('?');
    if (i > -1) {
      path = path.substring(0, i);
    }
    return path;
  }

  public static String decodePath(String path) throws DecodingException {
    try {
      return decode(path, UTF_8.displayName());
    } catch (UnsupportedEncodingException | IllegalArgumentException e) {
      throw new DecodingException(format("Unable to decode malformed url %s", path), e);
    }
  }

  public static byte[] multipartToBytes(MultipartHttpEntity entity, MediaType mediaType, String boundary,
                                        Map<String, String> additionalPartHeaders)
      throws IOException {
    try {
      List<BodyPart> bodyParts = new ArrayList<>();
      for (HttpPart httpPart : entity.getParts()) {
        final String subType = mediaType.getSubType();
        bodyParts.add(httpPartToBodyPart(httpPart, "mixed".equals(subType) ? "attachment" : subType, additionalPartHeaders));
      }

      MimeMultipart mimeMultipart = new MimeMultipart(new MultipartDataSource() {

        @Override
        public InputStream getInputStream() throws IOException {
          throw new UnsupportedOperationException();
        }

        @Override
        public OutputStream getOutputStream() throws IOException {
          throw new UnsupportedOperationException();
        }

        @Override
        public String getContentType() {
          return mediaType.toRfcString();
        }

        @Override
        public String getName() {
          throw new UnsupportedOperationException();
        }

        @Override
        public int getCount() {
          return bodyParts.size();
        }

        @Override
        public BodyPart getBodyPart(int index) throws MessagingException {
          return bodyParts.get(index);
        }

      });

      // Write the MimeMultipart to an in-memory buffer so callers can consume it
      // as a plain InputStream. MimeMultipart does not expose its raw bytes
      // directly & the only way is through the writeTo(OutputStream) method.
      ByteArrayOutputStream baos = new ByteArrayOutputStream();
      mimeMultipart.writeTo(baos);
      return baos.toByteArray();
    } catch (MessagingException e) {
      throw new IOException("Error processing multipart content", e);
    }
  }

  private static BodyPart httpPartToBodyPart(HttpPart httpPart, String subType, Map<String, String> additionalPartHeaders)
      throws MessagingException, IOException {
    MimeBodyPart bodyPart = new MimeBodyPart();
    byte[] contentBytes = toByteArray(httpPart.getInputStream());

    bodyPart
        .setDataHandler(new DataHandler(new ByteArrayDataSource(contentBytes, httpPart.getContentType())));

    final Collection<String> headerNames = httpPart.getHeaderNames();

    // Transfer common headers present on the HttpPart to the BodyPart so that
    // downstream consumers receive the exact same information.
    boolean contentTypeHeaderPresent = false;
    boolean contentDispositionHeaderPresent = false;
    for (String headerName : headerNames) {
      if (headerName.equalsIgnoreCase(CONTENT_TYPE)) {
        contentTypeHeaderPresent = true;
      }
      if (headerName.equalsIgnoreCase(CONTENT_DISPOSITION)) {
        contentDispositionHeaderPresent = true;
      }
      for (String headerValue : httpPart.getHeaders(headerName)) {
        bodyPart.addHeader(headerName, headerValue);
      }
    }
    if (!contentTypeHeaderPresent) {
      bodyPart.addHeader(CONTENT_TYPE, httpPart.getContentType());
    }
    if (!contentDispositionHeaderPresent && httpPart.getName() != null) {
      bodyPart.addHeader(CONTENT_DISPOSITION, subType + "; name=\"" + httpPart.getName() + "\"");
    }

    for (Entry<String, String> additionalHeader : additionalPartHeaders.entrySet()) {
      if (!headerNames.contains(additionalHeader.getKey())) {
        bodyPart.addHeader(additionalHeader.getKey(), additionalHeader.getValue());
      }
    }

    return bodyPart;
  }

  public static ByteArrayHttpEntity fromMultipartEntity(String contentType, MultipartHttpEntity entity,
                                                        Consumer<String> contentTypeConsumer,
                                                        Map<String, String> additionalPartHeaders)
      throws IOException {
    MediaType mediaType;
    String boundary;

    if (contentType == null) {
      boundary = generateBoundary();
      mediaType = MULTIPART_FORM_DATA.withParamaters(singletonMap(BOUNDARY, boundary));
      contentTypeConsumer.accept(mediaType.toRfcString());
    } else {
      mediaType = MediaType.parse(contentType);
      boundary = mediaType.getParameter(BOUNDARY);
      if (boundary == null) {
        boundary = generateBoundary();
        mediaType = mediaType.withParamaters(singletonMap(BOUNDARY, boundary));
        contentTypeConsumer.accept(mediaType.toRfcString());
      }
    }

    return new ByteArrayHttpEntity(multipartToBytes(entity, mediaType, boundary,
                                                    additionalPartHeaders));
  }

  private static String generateBoundary() {
    return "----MuleMultipart" + randomUUID().toString().replace("-", "");
  }

  public static Collection<HttpPart> parseMultipartContent(InputStream content, String contentType) throws IOException {
    MimeMultipart mimeMultipart = null;
    List<HttpPart> parts = Lists.newArrayList();

    try {
      mimeMultipart = new MimeMultipart(new ByteArrayDataSource(content, contentType));
    } catch (MessagingException e) {
      throw new IOException(e);
    }

    try {
      int partCount = mimeMultipart.getCount();

      for (int i = 0; i < partCount; i++) {
        BodyPart part = mimeMultipart.getBodyPart(i);

        String filename = part.getFileName();
        String partName = filename;
        String[] contentDispositions = part.getHeader(CONTENT_DISPOSITION);
        if (contentDispositions != null) {
          String contentDisposition = contentDispositions[0];
          if (contentDisposition.contains(NAME_ATTRIBUTE)) {
            partName = contentDisposition.substring(contentDisposition.indexOf(NAME_ATTRIBUTE) + NAME_ATTRIBUTE.length() + 2);
            partName = partName.substring(0, partName.indexOf("\""));
          }
        }

        if (partName == null && mimeMultipart.getContentType().contains(MULTIPART_RELATED.toString())) {
          String[] contentIdHeader = part.getHeader(CONTENT_ID);
          if (contentIdHeader != null && contentIdHeader.length > 0) {
            partName = contentIdHeader[0];
          }
        }

        HttpPart httpPart =
            new HttpPart(partName, filename, IOUtils.toByteArray(part.getInputStream()), part.getContentType(), part.getSize());

        Enumeration<Header> headers = part.getAllHeaders();

        while (headers.hasMoreElements()) {
          Header header = headers.nextElement();
          httpPart.addHeader(header.getName(), header.getValue());
        }
        parts.add(httpPart);
      }
    } catch (MessagingException e) {
      throw new IOException(e);
    }

    return parts;
  }

  /**
   * Normalize a path that may contains spaces, %20 or +.
   *
   * @param path path with encoded spaces or raw spaces
   * @return path with only spaces.
   */
  public static String normalizePathWithSpacesOrEncodedSpaces(String path) {
    return SPACE_ENTITY_OR_PLUS_SIGN_REGEX.matcher(path).replaceAll(WHITE_SPACE);
  }
}
