/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.client;

import static org.mule.functional.junit4.matchers.ThrowableCauseMatcher.hasCause;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.CREATED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.EXPECTATION_FAILED;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.UNAUTHORIZED;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.COOKIE;
import static org.mule.runtime.http.api.HttpHeaders.Names.EXPECT;
import static org.mule.runtime.http.api.client.auth.HttpAuthenticationType.BASIC;
import static org.mule.runtime.http.api.client.auth.HttpAuthenticationType.DIGEST;
import static org.mule.service.http.test.netty.utils.server.AuthenticationTestServer.DEFAULT_RESPONSE;
import static org.mule.tck.probe.PollingProber.probe;

import static java.lang.Integer.MAX_VALUE;
import static java.lang.String.format;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.concurrent.Executors.newWorkStealingPool;

import static io.netty.handler.codec.http.HttpHeaderNames.TRANSFER_ENCODING;
import static io.netty.handler.codec.http.HttpHeaderValues.CONTINUE;
import static io.netty.handler.codec.http.HttpMethod.GET;
import static io.netty.handler.codec.http.HttpMethod.POST;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.startsWith;
import static org.junit.Assert.assertThrows;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.client.auth.HttpAuthenticationType;
import org.mule.runtime.http.api.client.proxy.ProxyConfig;
import org.mule.runtime.http.api.client.ws.WebSocketCallback;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.entity.multipart.HttpPart;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.service.http.netty.impl.client.InetNoopAddressResolverGroup;
import org.mule.service.http.netty.impl.client.NettyHttpClient;
import org.mule.service.http.netty.impl.message.NettyHttpMessage;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.netty.tck.ExecutorRule;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.server.AuthenticationTestServer;
import org.mule.service.http.test.netty.utils.server.HardcodedResponseTcpServer;
import org.mule.service.http.test.netty.utils.server.TestHttpServer;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeoutException;

import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.ssl.SslContext;
import io.netty.resolver.AddressResolverGroup;
import io.netty.resolver.dns.RoundRobinDnsAddressResolverGroup;
import io.qameta.allure.Description;
import io.qameta.allure.Issue;
import org.apache.commons.io.IOUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

public class NettyHttpClientTestCase extends AbstractMuleTestCase {

  public static final String WRONG_USERNAME = "failing";
  public static final String WRONG_PASSWORD = "wrong";
  private static final int CONNECTION_IDLE_TIMEOUT = 5000;
  private static final String TEST_USERNAME = "testUsername";
  private static final String TEST_PASSWORD = "testPassword";
  private static final String TEST_DOMAIN = "ntlmDomain";

  @ClassRule
  public static ExecutorRule executorRule = new ExecutorRule();

  private HttpClient client;

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  @Rule
  public DynamicPort authServerPort = new DynamicPort("authServerPort");

  @Rule
  public DynamicPort hardcodedServerPort = new DynamicPort("hardcodedServerPort");

  @Rule
  public TestHttpServer testServer = new TestHttpServer("localhost", serverPort.getNumber(), false);

  @Rule
  public HardcodedResponseTcpServer hardcodedResponseTcpServer = new HardcodedResponseTcpServer(hardcodedServerPort.getNumber());

  @Rule
  public AuthenticationTestServer authTestServer =
      new AuthenticationTestServer(authServerPort.getNumber());

  private Latch serverLatch;

  @Before
  public void setUp() throws Exception {
    client = NettyHttpClient.builder()
        .withConnectionIdleTimeout(CONNECTION_IDLE_TIMEOUT)
        .withUsingPersistentConnections(true)
        .withIOTasksScheduler(executorRule.getExecutor())
        .build();
    client.start();

    testServer.addRequestHandler("/hello", (request, responseSender) -> {
      HttpEntity helloContent = new StringHttpEntity("Hello from server!");
      responseSender.responseReady(new ResponseWithoutHeaders(OK, helloContent), new NoOpResponseStatusCallback());
    }).start();
    serverLatch = new Latch();
    testServer.addRequestHandler("/waitForLatch", (request, responseSender) -> {
      try {
        serverLatch.await();
      } catch (InterruptedException e) {
        throw new RuntimeException(e);
      }
      HttpEntity helloContent = new StringHttpEntity("Hello from server!");
      responseSender.responseReady(new ResponseWithoutHeaders(OK, helloContent), new NoOpResponseStatusCallback());
    }).start();
  }

  @After
  public void tearDown() {
    client.stop();
  }

  @Test
  public void sendGetRequestToExistentPath() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContentAsString, is("Hello from server!"));
  }

  @Test
  public void communityEditionDoesNotSupportWebsockets() {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/irrelevant", serverPort.getNumber()))
        .method("GET")
        .build();
    WebSocketCallback mockCallback = mock(WebSocketCallback.class);
    UnsupportedOperationException exception =
        assertThrows(UnsupportedOperationException.class, () -> client.openWebSocket(httpRequest, "mockSocketID", mockCallback));
    assertThat(exception.getMessage(), containsString("WebSockets are only supported in Enterprise Edition"));
  }

  @Test
  public void sendGetRequestToNonExistentPath() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/not-existent", serverPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(HttpConstants.HttpStatus.NOT_FOUND.getStatusCode()));
  }

  @Test
  public void sendPostRequestToExistentPath() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/hello", serverPort.getNumber()))
        .method("POST")
        .entity(new StringHttpEntity("Hello from client!"))
        .build();
    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContentAsString, is("Hello from server!"));
  }

  @Test
  public void sendPostRequestToNonExistentPath() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/not-existent", serverPort.getNumber()))
        .method("POST")
        .entity(new StringHttpEntity("Hello from client"))
        .build();
    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(HttpConstants.HttpStatus.NOT_FOUND.getStatusCode()));
  }

  @Test
  public void connectionIdleTimeout() throws ExecutionException, InterruptedException {
    int shortTimeout = 500;
    // Use another client to execute faster.
    NettyHttpClient shortTimeoutClient = NettyHttpClient.builder()
        .withConnectionIdleTimeout(shortTimeout)
        .withUsingPersistentConnections(true)
        .build();
    shortTimeoutClient.start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/hello", hardcodedServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpResponse response = shortTimeoutClient.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    assertThat(hardcodedResponseTcpServer.acceptedCount(), is(1));
    probe(CONNECTION_IDLE_TIMEOUT + 20, 10, () -> hardcodedResponseTcpServer.acceptedCount() == 0);

    shortTimeoutClient.stop();
  }

  @Test
  public void persistentConnectionTest() throws ExecutionException, InterruptedException {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/hello", hardcodedServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpResponse response1 = client.sendAsync(httpRequest).get();
    assertThat(response1.getStatusCode(), is(OK.getStatusCode()));
    assertThat(hardcodedResponseTcpServer.acceptedCount(), is(1));

    // Give some time to the channel to go back to the pool. Otherwise, as it's all
    // non-blocking, we could get a new channel just because the old one wasn't offered
    // to the pool.
    Thread.sleep(500);

    HttpResponse response2 = client.sendAsync(httpRequest).get();
    assertThat(response2.getStatusCode(), is(OK.getStatusCode()));
    assertThat(hardcodedResponseTcpServer.acceptedCount(), is(1)); // Still 1!!!
  }

  @Test
  public void responseTimeout() {
    int tooShortTimeout = 30;

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/waitForLatch", serverPort.getNumber()))
        .build();

    CompletableFuture<HttpResponse> future =
        client.sendAsync(httpRequest, HttpRequestOptions.builder().responseTimeout(tooShortTimeout).build());
    ExecutionException executionException = assertThrows(ExecutionException.class, future::get);
    assertThat(executionException.getCause(), instanceOf(TimeoutException.class));

    serverLatch.release();
  }

  @Test
  public void sendNotEmptyBodyRequest() throws Exception {
    // TODO: move this to use the Netty Server when bug related to receiving streamed content is fixe
    // see W-15469458 (https://gus.lightning.force.com/lightning/r/ADM_Work__c/a07EE00001oEEIkYAO/view)
    String message = "Hello from client, this is a non streamed request";
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/basic/", authServerPort.getNumber()))
        .method("POST")
        .entity(new StringHttpEntity(message))
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(BASIC)
        .preemptive(true).username(TEST_USERNAME).password(TEST_PASSWORD).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    assertThat(authTestServer.getRequestBody(), is(message));
  }

  @Test
  public void sendStreamedRequest() throws Exception {
    // TODO: move this to use the Netty Server when bug related to receiving streamed content is fixed
    // see W-15469458 (https://gus.lightning.force.com/lightning/r/ADM_Work__c/a07EE00001oEEIkYAO/view)
    String message = "Hello from client, post request to existing path. This is an stream";
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/basic/", authServerPort.getNumber()))
        .method("POST")
        .entity(new InputStreamHttpEntity(new ByteArrayInputStream(message
            .getBytes(UTF_8))))
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(BASIC)
        .preemptive(true).username(TEST_USERNAME).password(TEST_PASSWORD).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    assertThat(authTestServer.getRequestBody(), is(message));
    String responseBody = getBodyFromResponse(response);
    assertThat(responseBody, is(DEFAULT_RESPONSE));
  }

  @Test
  public void sendBasicAuthRequest_preemptive() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/basic/", authServerPort.getNumber()))
        .method("POST")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(BASIC)
        .preemptive(true).username(TEST_USERNAME).password(TEST_PASSWORD).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
  }

  @Test
  public void sendBasicAuthRequest_preemptive_fails() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/basic/", authServerPort.getNumber()))
        .method("POST")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(BASIC)
        .preemptive(true).username(WRONG_USERNAME).password(WRONG_PASSWORD).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(UNAUTHORIZED.getStatusCode()));
  }

  @Test
  public void sendBasicAuthRequest_nonPreemptive() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/basic", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(BASIC)
        .preemptive(false).username(TEST_USERNAME).password(TEST_PASSWORD).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
  }

  @Test
  public void sendBasicAuthRequest_nonPreemptive_fails() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/basic", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(BASIC)
        .preemptive(false).username(WRONG_USERNAME).password(WRONG_PASSWORD).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(UNAUTHORIZED.getStatusCode()));
  }

  @Test
  public void sendDigestAuthRequest() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/digest", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(DIGEST)
        .username(TEST_USERNAME).password(TEST_PASSWORD).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
  }

  @Test
  public void sendDigestAuthRequest_fails() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/digest", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(DIGEST)
        .username(WRONG_USERNAME).password(WRONG_PASSWORD).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(UNAUTHORIZED.getStatusCode()));
  }

  @Test
  public void sendNtlmAuthRequest() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/ntlm", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.HttpNtlmAuthentication.builder()
        .type(HttpAuthenticationType.NTLM).username(TEST_USERNAME).password(TEST_PASSWORD).domain(TEST_DOMAIN).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
  }


  @Test
  @Issue("W-15631618")
  public void testMultipleWWWHeaders_whenBasicAuthenticationIsInvalid_thenServerShouldRespondWithMultipleWWWAuthenticationHeaders()
      throws Exception {

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/multi-auth", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();

    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(BASIC)
        .preemptive(true).username(WRONG_USERNAME)
        .password(WRONG_PASSWORD).build()).build();

    HttpResponse response = client.sendAsync(httpRequest, options).get();

    List<String> authenticateHeaders = response.getHeaders().getAll("www-authenticate");
    assertThat(authenticateHeaders, hasItem(containsString("Basic realm=\"Test\"")));
    assertThat(authenticateHeaders, hasItem(containsString("NTLM")));
    assertThat(authenticateHeaders, hasItem(startsWith("Digest realm=\"Test\"")));
    assertThat(response.getStatusCode(), is(401));
  }

  @Test
  @Issue("W-15631618")
  public void testMultipleWWWHeaders_whenDigestAuthenticationIsInvalid_thenServerShouldRespondWithMultipleWWWAuthenticationHeaders()
      throws Exception {

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/multi-auth", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(DIGEST)
        .username(WRONG_USERNAME).password(WRONG_PASSWORD)
        .build()).build();

    HttpResponse response = client.sendAsync(httpRequest, options).get();

    List<String> authenticateHeaders = response.getHeaders().getAll("www-authenticate");
    assertThat(authenticateHeaders, hasItem(containsString("Basic realm=\"Test\"")));
    assertThat(authenticateHeaders, hasItem(containsString("NTLM")));
    assertThat(authenticateHeaders, hasItem(startsWith("Digest realm=\"Test\"")));
    assertThat(response.getStatusCode(), is(401));
  }

  @Test
  @Issue("W-15631618")
  public void testMultipleWWWHeaders_whenAuthenticationIsValid_thenServerShouldRespondWith200() throws Exception {

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/multi-auth", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();

    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.HttpNtlmAuthentication.builder()
        .type(
              BASIC)
        .username(TEST_USERNAME).password(TEST_PASSWORD)
        .domain(TEST_DOMAIN).build()).build();
    HttpResponse response = client.sendAsync(httpRequest, options).get();
    assertThat(response.getStatusCode(), is(200));
  }


  @Test
  public void sendEmptyBodyAndGet_noContentLength() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/emptyBody", serverPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    testServer.removeAllHandlers();
    testServer.addRequestHandler("/emptyBody", (request, responseSender) -> {
      NettyHttpMessage nettyHttpMessage = (NettyHttpMessage) request.getRequest();
      assertThat("Content-Length is present when it should be missing",
                 nettyHttpMessage.containsHeader(CONTENT_LENGTH), is(false));
      responseSender.responseReady(new ResponseWithoutHeaders(OK, new EmptyHttpEntity()), new NoOpResponseStatusCallback());
    });
    client.sendAsync(httpRequest).get();
  }

  @Test
  public void sendEmptyBodyAndPost_contentLengthPresentAndZero() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/emptyBody", serverPort.getNumber()))
        .method("POST")
        .entity(new EmptyHttpEntity())
        .build();
    testServer.removeAllHandlers();
    testServer.addRequestHandler("/emptyBody", (request, responseSender) -> {
      NettyHttpMessage nettyHttpMessage = (NettyHttpMessage) request.getRequest();
      assertThat("Content-Length is missing when it should be present", nettyHttpMessage.containsHeader(CONTENT_LENGTH),
                 is(true));
      assertThat("Content-Length should be 0", Integer.valueOf(nettyHttpMessage.getHeaderValue(CONTENT_LENGTH)), is(0));
      responseSender.responseReady(new ResponseWithoutHeaders(OK, new EmptyHttpEntity()), new NoOpResponseStatusCallback());
    });
    client.sendAsync(httpRequest).get();
  }

  @Test
  public void readContentAsSoonFirstChunkReturned() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/basic", authServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    HttpRequestOptions options = HttpRequestOptions.builder().authentication(HttpAuthentication.builder().type(BASIC)
        .preemptive(false).username(TEST_USERNAME).password(TEST_PASSWORD).build()).build();
    client.sendAsync(httpRequest, options).handle((response, exception) -> {
      try {
        return getBodyFromResponse(response);
      } catch (IOException | InterruptedException e) {
        throw new CompletionException(e);
      }
    }).whenComplete((result, exception) -> {
      assertThat(result, is(DEFAULT_RESPONSE));
    }).get();
  }

  @Test
  public void maxConnections() throws Exception {
    client = NettyHttpClient.builder()
        .withConnectionIdleTimeout(CONNECTION_IDLE_TIMEOUT)
        .withUsingPersistentConnections(true)
        .withMaxConnections(1)
        .build();
    client.start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/hello", hardcodedServerPort.getNumber()))
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    client.sendAsync(httpRequest).get();
    client.sendAsync(httpRequest).get();
  }

  @Test
  public void expectationFailed() throws Exception {
    testServer.addRequestHandler("/headerExpectationFailed", (request, responseSender) -> {
      responseSender.responseReady(new ResponseWithoutHeaders(EXPECTATION_FAILED, new EmptyHttpEntity()),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/headerExpectationFailed", serverPort.getNumber()))
        .addHeader(EXPECT, "wrongValue")
        .method("GET")
        .entity(new EmptyHttpEntity()).build();

    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(EXPECTATION_FAILED.getStatusCode()));
  }

  @Test
  public void requestBodyNotSendOn417() throws Exception {
    testServer.addRequestHandler("/headerExpectationFailed", (request, responseSender) -> {
      responseSender.responseReady(new ResponseWithoutHeaders(EXPECTATION_FAILED, new EmptyHttpEntity()),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/headerExpectationFailed", serverPort.getNumber()))
        .addHeader(EXPECT, "wrongValue")
        .method("GET")
        .entity(new InputStreamHttpEntity(new InputStream() {

          @Override
          public int read() throws IOException {
            throw new IOException("Payload should not be consumed");
          }
        })).build();

    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(EXPECTATION_FAILED.getStatusCode()));
  }

  @Test
  public void requestBodySentAfter100Continue() throws Exception {
    String serverResponse = "Hello from server";
    testServer.addRequestHandler("/100Continue", (request, responseSender) -> {
      responseSender.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity(serverResponse)),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/100Continue", serverPort.getNumber()))
        .addHeader(EXPECT, CONTINUE.toString())
        .method("GET")
        .entity(new StringHttpEntity("Hello From Client"))
        .build();

    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    assertThat(getBodyFromResponse(response), is(serverResponse));
  }


  @Test
  @Issue("W-15631467")
  public void testExpectContinueBehavior() throws Exception {
    testServer.addRequestHandler("/expectContinue", (request, responseSender) -> {
      if (request.getRequest().getHeaders().containsKey("Expect")) {
        responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.CONTINUE, new EmptyHttpEntity()),
                                     new NoOpResponseStatusCallback());
      }

      HttpEntity responseContent = new StringHttpEntity("Request processed successfully!");
      responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.OK, responseContent),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/expectContinue", serverPort.getNumber()))
        .method("POST")
        .addHeader("Expect", "100-continue")
        .entity(new StringHttpEntity("This is the body of the request."))
        .build();

    HttpResponse response = client.sendAsync(httpRequest).get();

    assertThat(response.getStatusCode(), is(HttpURLConnection.HTTP_OK));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContentAsString, is("Request processed successfully!"));
  }

  @Test
  @Issue("W-16935919")
  public void testExpectContinueBehaviorWithTransferEncodingChunked() throws Exception {
    testServer.addRequestHandler("/expectContinue", NettyHttpClientTestCase::consumeContentAndRespond).start();

    int chunkedPayloadSize = 1 << 26;

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/expectContinue", serverPort.getNumber()))
        .method("POST")
        .addHeader("Expect", "100-continue")
        .addHeader("transfer-encoding", "chunked")
        .entity(new InputStreamHttpEntity(new DummyInputStream(chunkedPayloadSize)))
        .build();

    HttpResponse response = client.sendAsync(httpRequest).get();

    assertThat(response.getStatusCode(), is(HttpURLConnection.HTTP_OK));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContentAsString, equalTo("Request length was: " + chunkedPayloadSize));
  }

  @Test
  @Issue("W-18706728")
  @Description("Validates that uri that requires DNS resolution is correctly resolved and response is correctly retrieved")
  public void testDnsResolution() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri("https://www.salesforce.com/")
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();

    HttpResponse response = client.sendAsync(httpRequest).get();
    assertThat(response.getStatusCode(), is(HttpURLConnection.HTTP_OK));
  }

  @Test
  @Issue("W-17142270")
  public void testExpectContinueBehaviourWithNonStreamingBody() throws Exception {
    CountDownLatch latch = new CountDownLatch(1);
    String bodyContent = "This is the body of the streaming request.";

    HttpEntity requestEntity = new HttpEntity() {

      @Override
      public boolean isStreaming() {
        return false;
      }

      @Override
      public boolean isComposed() {
        return false;
      }

      @Override
      public InputStream getContent() {
        return new ByteArrayInputStream(bodyContent.getBytes(StandardCharsets.UTF_8));
      }

      @Override
      public byte[] getBytes() {
        return bodyContent.getBytes(StandardCharsets.UTF_8);
      }

      @Override
      public Collection<HttpPart> getParts() {
        return List.of();
      }

      @Override
      public Optional<Long> getLength() {
        return Optional.of((long) bodyContent.length());
      }
    };

    testServer.addRequestHandler("/100Continue", (request, responseSender) -> {
      if (request.getRequest().getHeaders().containsKey("Expect")) {
        responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.CONTINUE, new EmptyHttpEntity()),
                                     new NoOpResponseStatusCallback());
      }

      HttpEntity responseContent = new StringHttpEntity("Request processed successfully!");
      responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.OK, responseContent),
                                   new NoOpResponseStatusCallback());

      // Signal to show that processing is complete
      latch.countDown();
    }).start();

    HttpRequest httpRequest =
        HttpRequest.builder().uri(format("http://localhost:%d/100Continue", serverPort.getNumber())).method("POST")
            .addHeader("Expect", "100-continue").entity(requestEntity)
            .build();

    CompletableFuture<HttpResponse> responseFuture = client.sendAsync(httpRequest);

    HttpResponse response = responseFuture.get();

    latch.await();

    assertThat(response.getStatusCode(), is(HttpURLConnection.HTTP_OK));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(responseContentAsString, is("Request processed successfully!"));
  }

  @Test
  @Issue("W-16935919")
  public void testExpectContinueBehaviourWithStreamingBody() throws Exception {
    CountDownLatch latch = new CountDownLatch(1);
    String bodyContent = "This is the body of the request.";
    InputStream bodyStream = new ByteArrayInputStream(bodyContent.getBytes(StandardCharsets.UTF_8));

    HttpEntity requestEntity = new HttpEntity() {

      @Override
      public boolean isStreaming() {
        return true;
      }

      @Override
      public boolean isComposed() {
        return false;
      }

      @Override
      public InputStream getContent() {
        return bodyStream;
      }

      @Override
      public byte[] getBytes() {
        return new byte[0];
      }

      @Override
      public Collection<HttpPart> getParts() {
        return List.of();
      }

      @Override
      public Optional<Long> getLength() {
        return Optional.empty();
      }
    };

    testServer.addRequestHandler("/100Continue", (request, responseSender) -> {
      if (request.getRequest().getHeaders().containsKey("Expect")) {
        responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.CONTINUE, new EmptyHttpEntity()),
                                     new NoOpResponseStatusCallback());
      }

      HttpEntity responseContent = new StringHttpEntity("Request processed successfully!");
      responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.OK, responseContent),
                                   new NoOpResponseStatusCallback());

      // Signal to show that processing is complete
      latch.countDown();
    }).start();

    HttpRequest httpRequest =
        HttpRequest.builder().uri(format("http://localhost:%d/100Continue", serverPort.getNumber())).method("POST")
            .addHeader("Expect", "100-continue").entity(requestEntity)
            .build();

    CompletableFuture<HttpResponse> responseFuture = client.sendAsync(httpRequest);

    HttpResponse response = responseFuture.get();

    latch.await();

    assertThat(response.getStatusCode(), is(HttpURLConnection.HTTP_OK));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), StandardCharsets.UTF_8);
    assertThat(responseContentAsString, is("Request processed successfully!"));
  }

  @Test
  @Issue("W-15631467")
  public void testClientClosesConnectionAfter100Continue() throws Exception {
    testServer.addRequestHandler("/expectContinue", (request, responseSender) -> {
      if (request.getRequest().getHeaders().containsKey("Expect")) {
        responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.CONTINUE, new EmptyHttpEntity()),
                                     new NoOpResponseStatusCallback());
      }

      // Wait for the body, but the client will not send it
      HttpEntity responseContent = new StringHttpEntity("This message should not be sent!");
      responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.OK, responseContent),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/expectContinue", serverPort.getNumber()))
        .method("POST")
        .addHeader("Expect", "100-continue")
        .entity(new StringHttpEntity("This is the body of the request."))
        .build();

    // Send the request but simulate a scenario where the client doesn't send the body
    CompletableFuture<HttpResponse> responseFuture = client.sendAsync(httpRequest);

    // Simulate the client closing the connection after receiving the 100 Continue response
    responseFuture.cancel(true);

    // Now the server should detect that the connection was closed and not send an OK response
    try {
      responseFuture.get();
      fail("Expected the client to close the connection and not receive a response");
    } catch (CancellationException e) {
    }
  }

  @Test
  @Issue("W-15631467")
  public void testServerIgnoresExpect100Continue() throws Exception {
    // Server responds with 200 OK directly without sending 100 Continue
    testServer.addRequestHandler("/expectContinue", (request, responseSender) -> {
      HttpEntity responseContent = new StringHttpEntity("Server ignored 100-continue.");
      responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.OK, responseContent),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/expectContinue", serverPort.getNumber()))
        .method("POST")
        .addHeader("Expect", "100-continue")
        .entity(new StringHttpEntity("This is the body of the request."))
        .build();

    // Send the request and wait for the response
    HttpResponse response = client.sendAsync(httpRequest).get();

    // Verify the response (server ignored the Expect header)
    assertThat(response.getStatusCode(), is(HttpURLConnection.HTTP_OK));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContentAsString, is("Server ignored 100-continue."));
  }

  @Test
  @Issue("W-15631467")
  public void testServerRejectsWith417ExpectationFailed() throws Exception {
    // Server responds with 417 Expectation Failed
    testServer.addRequestHandler("/expectContinue", (request, responseSender) -> {
      HttpEntity responseContent = new StringHttpEntity("Expectation failed.");
      responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.EXPECTATION_FAILED, responseContent),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/expectContinue", serverPort.getNumber()))
        .method("POST")
        .addHeader("Expect", "100-continue")
        .entity(new StringHttpEntity("This is the body of the request."))
        .build();

    HttpResponse response = client.sendAsync(httpRequest).get();

    // Verify the server rejected the Expect: 100-continue request
    assertThat(response.getStatusCode(), is(EXPECTATION_FAILED.getStatusCode()));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContentAsString, is("Expectation failed."));
  }

  @Test
  @Issue("W-15631467")
  public void testServerSendsErrorAfter100Continue() throws Exception {
    // Server first sends 100 Continue and then 500 Internal Server Error
    testServer.addRequestHandler("/expectContinue", (request, responseSender) -> {
      if (request.getRequest().getHeaders().containsKey("Expect")) {
        responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.CONTINUE, new EmptyHttpEntity()),
                                     new NoOpResponseStatusCallback());
      }

      HttpEntity errorContent = new StringHttpEntity("Internal Server Error.");
      responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR, errorContent),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = HttpRequest.builder()
        .uri(format("http://localhost:%d/expectContinue", serverPort.getNumber()))
        .method("POST")
        .addHeader("Expect", "100-continue")
        .entity(new StringHttpEntity("This is the body of the request."))
        .build();

    HttpResponse response = client.sendAsync(httpRequest).get();

    assertThat(response.getStatusCode(), is(HttpURLConnection.HTTP_INTERNAL_ERROR));
    String responseContentAsString = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContentAsString, is("Internal Server Error."));
  }

  @Test
  @Issue("W-15631467")
  public void testExpectContinueWithNewLineSeparator() throws Exception {
    testServer.addRequestHandler("/expectContinue", (request, responseSender) -> {
      if (request.getRequest().getHeaders().containsKey("Expect")) {
        responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.CONTINUE, new EmptyHttpEntity()),
                                     new NoOpResponseStatusCallback());
      }
      HttpEntity responseContent = new StringHttpEntity("Request processed successfully!");
      responseSender.responseReady(new ResponseWithoutHeaders(HttpConstants.HttpStatus.OK, responseContent),
                                   new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequestNewLine = HttpRequest.builder()
        .uri(format("http://localhost:%d/expectContinue", serverPort.getNumber()))
        .method("POST")
        .addHeader("Expect", "100-continue \n")
        .entity(new StringHttpEntity("This is the body of the request using \\n line separator."))
        .build();

    HttpResponse responseNewLine = client.sendAsync(httpRequestNewLine).get();
    assertThat(responseNewLine.getStatusCode(), is(HttpURLConnection.HTTP_OK));
    String responseContentNewLine = IOUtils.toString(responseNewLine.getEntity().getContent(), UTF_8);
    assertThat(responseContentNewLine, is("Request processed successfully!"));
  }

  @Test
  @Issue("W-15631644")
  public void sendsSeveralLargeRequestStreamInParallel() throws ExecutionException, InterruptedException, IOException {
    int requestLength = MAX_VALUE / 2; // Divide by two to avoid overflows
    sendMultipleStreamingRequestsInParallel(16, requestLength);
  }

  @Test
  @Issue("W-15631644")
  public void sendsManyMediumRequestStreamInParallel() throws ExecutionException, InterruptedException, IOException {
    sendMultipleStreamingRequestsInParallel(100, 1 << 25);
  }

  @Test
  @Issue("W-17051925")
  public void testResolverGroupWithProxyConfiguredAsTunnel() {
    ProxyConfig proxyConfig = mockProxyConfig();
    SslContext sslContext = mock(SslContext.class);

    NettyHttpClient clientWithProxyAndSSL = NettyHttpClient.builder()
        .withProxyConfig(proxyConfig)
        .withSslContext(sslContext)
        .withConnectionIdleTimeout(CONNECTION_IDLE_TIMEOUT)
        .withUsingPersistentConnections(true)
        .build();
    clientWithProxyAndSSL.start();

    // Get the resolver group
    AddressResolverGroup<InetSocketAddress> resolverGroup = clientWithProxyAndSSL.getResolverGroup();
    // Assert that the resolverGroup is the InetNoopAddressResolverGroup
    assertThat(resolverGroup, is(InetNoopAddressResolverGroup.INSTANCE));

    clientWithProxyAndSSL.stop();
  }

  @Test
  @Issue("W-17051925")
  public void testResolverGroupWithoutProxy() {
    NettyHttpClient clientWithoutProxy = NettyHttpClient.builder()
        .withConnectionIdleTimeout(CONNECTION_IDLE_TIMEOUT)
        .withUsingPersistentConnections(true)
        .build();
    clientWithoutProxy.start();

    // Get the resolver group
    AddressResolverGroup<InetSocketAddress> resolverGroup = clientWithoutProxy.getResolverGroup();
    // Assert that the resolverGroup is an instance of RoundRobinDnsAddressResolverGroup
    assertThat(resolverGroup, instanceOf(RoundRobinDnsAddressResolverGroup.class));

    clientWithoutProxy.stop();
  }

  @Test
  @Issue("W-15631628")
  public void sendGetRequestRedirected() throws Exception {

    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {

      HttpResponse finalResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17908187")
  public void sendGetRequestRedirectedWithInitialCookieSent() throws Exception {

    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {

      assertThat(request.getRequest().getHeaders().containsKey(COOKIE), is(true));
      assertThat(request.getRequest().getHeaders().get(COOKIE).contains("cookie1=value1"), is(true));
      HttpResponse finalResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpRequest request = HttpRequest.builder()
        .uri(format("http://localhost:%d/redirect", serverPort.getNumber()))
        .method(HttpConstants.Method.GET.name())
        .entity(new EmptyHttpEntity())
        .addHeader(COOKIE, "cookie1=value1")
        .build();

    HttpResponse response =
        client.sendAsync(request,
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17847592")
  public void sendPostRequestRedirectShouldBeGetRequestWhen302AndBodyShouldBeRemoved() throws Exception {

    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      NettyHttpMessage httpMessage = (NettyHttpMessage) request.getRequest();
      assertThat(httpMessage.getHeaderValue(CONTENT_LENGTH), is("31"));
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      assertThat(request.getRequest().getMethod(), is(GET.name()));
      NettyHttpMessage nettyHttpMessage = (NettyHttpMessage) request.getRequest();
      assertThat("Content-Length should be removed",
                 nettyHttpMessage.containsHeader(String.valueOf(TRANSFER_ENCODING)), is(false));
      assertThat(nettyHttpMessage.getHeaderValue(CONTENT_LENGTH), is("0"));
      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client
            .sendAsync(buildHttpRequest(POST, "http://localhost:%d/redirect",
                                        new StringHttpEntity("I should be removed on redirect")),
                       HttpRequestOptions.builder().followsRedirect(true).sendBodyAlways(false).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17847592")
  public void sendPostRequestRedirectShouldBeGetRequestWhen302AndBodyShouldBePreserved() throws Exception {

    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      NettyHttpMessage httpMessage = (NettyHttpMessage) request.getRequest();
      assertThat(httpMessage.getHeaderValue(CONTENT_LENGTH), is("33"));
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      assertThat(request.getRequest().getMethod(), is(GET.name()));
      NettyHttpMessage nettyHttpMessage = (NettyHttpMessage) request.getRequest();
      assertThat(nettyHttpMessage.getHeaderValue(CONTENT_LENGTH), is("33"));
      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client
            .sendAsync(buildHttpRequest(POST, "http://localhost:%d/redirect",
                                        new StringHttpEntity("I should be preserved on redirect")),
                       HttpRequestOptions.builder().followsRedirect(true).sendBodyAlways(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17847592")
  public void sendPostRequestRedirectShouldBeGetRequestWhen303() throws Exception {

    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      NettyHttpMessage httpMessage = (NettyHttpMessage) request.getRequest();
      assertThat(httpMessage.getHeaderValue(CONTENT_LENGTH), is("31"));
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(303).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      assertThat(request.getRequest().getMethod(), is(GET.name()));

      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client
            .sendAsync(buildHttpRequest(POST, "http://localhost:%d/redirect",
                                        new StringHttpEntity("I should be removed on redirect")),
                       HttpRequestOptions.builder().followsRedirect(true).sendBodyAlways(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17755418")
  public void sendGetRequestRedirectedSeveralTimes() throws Exception {

    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-1");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-1", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-2");
      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 1 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-2", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-3");
      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 2 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-3", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 3 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(HttpMethod.GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();


    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17755418")
  public void sendGetRequestRedirectedMoreThanMaxRedirectedLimit() {

    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-1");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-1", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-2");
      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 1 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-2", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-3");
      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 2 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-3", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-4");
      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 3 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-4", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-5");
      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 4 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-5", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 5 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpRequest httpRequest = buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity());

    CompletableFuture<HttpResponse> future =
        client.sendAsync(httpRequest, HttpRequestOptions.builder().followsRedirect(true).build());
    ExecutionException executionException = assertThrows(ExecutionException.class, future::get);
    assertThat(executionException.getCause(), instanceOf(MuleRuntimeException.class));
  }

  @Test
  @Issue("W-15631628")
  public void sendGetRequestNotRedirected() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "http://www.example.com/");
      HttpResponse redirectResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Redirecting"))
          .headers(headers)
          .statusCode(302)
          .build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(false).build())
            .get();

    assertThat(response.getStatusCode(), is(302));
    assertThat(response.getHeaders().get("Location"), is("http://www.example.com/"));
  }

  @Test
  @Issue("W-17405565")
  public void sendGetRequestShouldRemoveSensitiveHeaders() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      headers.put("Authorization", "Some-value");
      HttpResponse redirectResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Redirecting"))
          .headers(headers)
          .statusCode(302)
          .build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Authorization"), is(false));

      HttpResponse finalResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Final Destination Reached"))
          .statusCode(200)
          .build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectPropagatesCookieWithExpiryDateInFuture() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      headers.put("Set-Cookie", "ValidCookie=ValidValue; Expires=Sun, 09 Jun 2030 10:18:14 GMT; Path=/");
      HttpResponse redirectResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Redirecting"))
          .headers(headers)
          .statusCode(302)
          .build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(true));
      assertThat(requestHeaders.get("Cookie").contains("ValidCookie=ValidValue"), is(true));
      HttpResponse finalResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Final Destination Reached"))
          .statusCode(200)
          .build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectPropagatesCookieWithMaxAgeInFuture() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      headers.put("Set-Cookie", "ValidCookie=ValidValue; Max-Age=34560000; Path=/");
      HttpResponse redirectResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Redirecting"))
          .headers(headers)
          .statusCode(302)
          .build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(true));
      assertThat(requestHeaders.get("Cookie").contains("ValidCookie=ValidValue"), is(true));
      HttpResponse finalResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Final Destination Reached"))
          .statusCode(200)
          .build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectRemovesExpiredCookies() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      headers.put("Set-Cookie", "ExpiredCookie=ExpiredValue; Expires=Thu, 01 Jan 1970 00:00:01 GMT; Path=/");
      HttpResponse redirectResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Redirecting"))
          .headers(headers)
          .statusCode(302)
          .build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(false));
      HttpResponse finalResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Final Destination Reached"))
          .statusCode(200)
          .build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectRemovesExpiredCookiesWithMaxAgeNegative() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      headers.put("Set-Cookie", "ExpiredCookie=ExpiredValue; Max-Age=-1; Path=/");
      HttpResponse redirectResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Redirecting"))
          .headers(headers)
          .statusCode(302)
          .build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(false));
      HttpResponse finalResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Final Destination Reached"))
          .statusCode(200)
          .build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectRemovesExpiredCookiesWithMaxAgeZero() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      headers.put("Set-Cookie", "ExpiredCookie=ExpiredValue; Max-Age=0; Path=/");
      HttpResponse redirectResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Redirecting"))
          .headers(headers)
          .statusCode(302)
          .build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(false));
      HttpResponse finalResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Final Destination Reached"))
          .statusCode(200)
          .build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectRemovesExpiredCookiesWithMaxAgeNegativeButExpiresInTheFuture() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/final");
      headers.put("Set-Cookie",
                  "ActuallyExpiredCookie=ActuallyExpiredValue; Max-Age=-1; Expires=Thu, 01 Jan 2070 00:00:01 GMT; Path=/");
      HttpResponse redirectResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Redirecting"))
          .headers(headers)
          .statusCode(302)
          .build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(false));

      HttpResponse finalResponse = HttpResponse.builder()
          .entity(new StringHttpEntity("Final Destination Reached"))
          .statusCode(200)
          .build();
      responseSender.responseReady(finalResponse, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectEachServerRespondsWithSetCookie() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-1");
      headers.put("Set-Cookie", "abc=123");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-1", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "final");
      headers.put("Set-Cookie", "abc1=1231");

      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 2 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {

      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(true));
      assertThat(requestHeaders.get("Cookie").contains("abc=123; abc1=1231"), is(true));
      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectEachServerIncrementsTheCookieValue() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-1");
      headers.put("Set-Cookie", "abc=1");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-1", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "final");
      headers.put("Set-Cookie", "abc=2");

      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 2 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {

      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(true));
      assertThat(requestHeaders.get("Cookie").contains("abc=2"), is(true));
      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17908187")
  public void sendGetRequestOnRedirectEachServerIncrementsTheCookieValueAndInitialCookieDoesNotGetsUpdated() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-1");
      headers.put("Set-Cookie", "abc=1");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-1", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "final");
      headers.put("Set-Cookie", "abc=2");

      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 2 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {

      assertThat(request.getRequest().getHeaders().containsKey(COOKIE), is(true));
      assertThat(request.getRequest().getHeaders().get(COOKIE).contains("abc=2"), is(true));
      assertThat(request.getRequest().getHeaders().get(COOKIE).contains("session=123"), is(true));
      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpRequest request = HttpRequest.builder()
        .uri(format("http://localhost:%d/redirect", serverPort.getNumber()))
        .method(GET.name())
        .entity(new EmptyHttpEntity())
        .addHeaders(COOKIE, List.of("session=123"))
        .build();

    HttpResponse response =
        client.sendAsync(request,
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17908187")
  public void sendGetRequestOnRedirectEachServerIncrementsTheCookieValueAndInitialCookieGetsUpdated() throws Exception {

    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-1");
      headers.put("Set-Cookie", "session=456");
      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-1", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "final");
      headers.put("Set-Cookie", "abc=2");

      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 2 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {
      assertThat(request.getRequest().getHeaders().containsKey(COOKIE), is(true));
      assertThat(request.getRequest().getHeaders().get(COOKIE).contains("abc=2"), is(true));
      assertThat(request.getRequest().getHeaders().get(COOKIE).contains("key=123"), is(true));
      assertThat(request.getRequest().getHeaders().get(COOKIE).contains("session=456"), is(true));

      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpRequest request = HttpRequest.builder()
        .uri(format("http://localhost:%d/redirect", serverPort.getNumber()))
        .method(GET.name())
        .entity(new EmptyHttpEntity())
        .addHeaders(COOKIE, List.of("abc=0", "key=123"))
        .build();

    HttpResponse response =
        client.sendAsync(request,
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-17773140")
  public void sendGetRequestOnRedirectOnlyFirstServerRespondsWithSetCookie() throws Exception {
    testServer.addRequestHandler("/redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/temp-1");
      headers.put("Set-Cookie", "key=value");

      HttpResponse redirectResponse =
          HttpResponse.builder().entity(new StringHttpEntity("Redirecting")).headers(headers).statusCode(302).build();
      responseSender.responseReady(redirectResponse, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/temp-1", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "final");

      HttpResponse response =
          HttpResponse.builder().entity(new StringHttpEntity("Temp 2 Destination Reached")).headers(headers).statusCode(302)
              .build();
      responseSender.responseReady(response, new NoOpResponseStatusCallback());
    }).start();

    testServer.addRequestHandler("/final", (request, responseSender) -> {

      MultiMap<String, String> requestHeaders = request.getRequest().getHeaders();
      assertThat(requestHeaders.containsKey("Cookie"), is(true));
      assertThat(requestHeaders.get("Cookie").contains("key=value"), is(true));
      HttpResponse temp1Response =
          HttpResponse.builder().entity(new StringHttpEntity("Final Destination Reached")).statusCode(200).build();
      responseSender.responseReady(temp1Response, new NoOpResponseStatusCallback());
    }).start();

    HttpResponse response =
        client.sendAsync(buildHttpRequest(GET, "http://localhost:%d/redirect", new EmptyHttpEntity()),
                         HttpRequestOptions.builder().followsRedirect(true).build())
            .get();

    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    String responseContent = IOUtils.toString(response.getEntity().getContent(), UTF_8);
    assertThat(responseContent, is("Final Destination Reached"));
  }

  @Test
  @Issue("W-19730211")
  public void sendGetRequestToNonExistentUrl() throws Exception {
    HttpRequest httpRequest = HttpRequest.builder()
        .uri("http://salesfooooorceeeeeeee.com")
        .method("GET")
        .entity(new EmptyHttpEntity())
        .build();
    ExecutionException thrown = assertThrows(ExecutionException.class, () -> client.sendAsync(httpRequest).get());
    assertThat(thrown, hasCause(instanceOf(ConnectionException.class)));
  }

  @Test
  public void noRedirectWhen201WithLocation() throws Exception {
    var requestHandlerMgr = testServer.addRequestHandler("/no-redirect", (request, responseSender) -> {
      MultiMap<String, String> headers = new MultiMap<>();
      headers.put("Location", "/not-existent");
      HttpResponse responseWithLocation = HttpResponse.builder()
          .statusCode(201)
          .headers(headers)
          .entity(new EmptyHttpEntity())
          .build();
      responseSender.responseReady(responseWithLocation, new NoOpResponseStatusCallback());
    });
    requestHandlerMgr.start();

    var request = buildHttpRequest(GET,
                                   "http://localhost:%d/no-redirect",
                                   new EmptyHttpEntity());
    var optionsWithRedirectEnabled = HttpRequestOptions.builder()
        .followsRedirect(true)
        .build();
    var response = client.send(request, optionsWithRedirectEnabled);

    assertThat(response.getStatusCode(), is(CREATED.getStatusCode()));

    requestHandlerMgr.stop();
  }

  private HttpRequest buildHttpRequest(HttpMethod method, String path, HttpEntity entity) {
    return HttpRequest.builder()
        .uri(format(path, serverPort.getNumber()))
        .method(method.name())
        .entity(entity)
        .build();
  }

  private ProxyConfig mockProxyConfig() {
    ProxyConfig proxyConfig = mock(ProxyConfig.class);
    when(proxyConfig.getHost()).thenReturn("proxyHost");
    when(proxyConfig.getPort()).thenReturn(8080);
    return proxyConfig;
  }

  private void sendMultipleStreamingRequestsInParallel(int requestsNumber, int requestsLength)
      throws InterruptedException, ExecutionException, IOException {
    ExecutorService executor = newWorkStealingPool(requestsNumber);

    testServer.addRequestHandler("/countBytes", (request, responseSender) -> {
      executor.submit(() -> consumeContentAndRespond(request, responseSender));
    }).start();

    List<CompletableFuture<HttpResponse>> futureList = new ArrayList<>(requestsNumber);
    for (int i = 0; i < requestsNumber; ++i) {
      HttpRequest httpRequest = HttpRequest.builder()
          .uri(format("http://localhost:%d/countBytes", serverPort.getNumber()))
          .method("POST")
          .entity(new InputStreamHttpEntity(new DummyInputStream(requestsLength)))
          .build();

      futureList.add(client.sendAsync(httpRequest));
    }

    for (CompletableFuture<HttpResponse> future : futureList) {
      HttpResponse response = future.get();
      assertThat(response.getStatusCode(), is(OK.getStatusCode()));
      assertThat(getBodyFromResponse(response), is("Request length was: " + requestsLength));
    }

    executor.shutdown();
  }

  private static void consumeContentAndRespond(HttpRequestContext request, HttpResponseReadyCallback responseSender) {
    InputStream content = request.getRequest().getEntity().getContent();
    boolean closed = false;
    byte[] buf = new byte[1024];
    int totalReadBytes = 0;
    while (!closed) {
      try {
        int readBytes = content.read(buf);
        if (readBytes == -1) {
          closed = true;
        } else {
          totalReadBytes += readBytes;
        }
      } catch (IOException e) {
        closed = true;
      }
    }
    responseSender.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity("Request length was: " + totalReadBytes)),
                                 new NoOpResponseStatusCallback());
  }

  private String getBodyFromResponse(HttpResponse response) throws IOException, InterruptedException {
    try (InputStream responseInputStream = response.getEntity().getContent()) {
      return IOUtils.toString(responseInputStream, UTF_8);
    }
  }

  private static final class DummyInputStream extends InputStream {

    private final int streamLength;
    private int alreadyRead;

    private DummyInputStream(int length) {
      this.streamLength = length;
      this.alreadyRead = 0;
    }

    @Override
    public synchronized int read() throws IOException {
      if (alreadyRead == streamLength) {
        return -1;
      }

      alreadyRead += 1;
      return 0;
    }

    @Override
    public synchronized int read(byte[] b, int off, int len) throws IOException {
      if (alreadyRead == streamLength) {
        // No more data
        return -1;
      }

      if (alreadyRead + len > streamLength) {
        // Not enough data
        int available = streamLength - alreadyRead;
        alreadyRead = streamLength;
        return available;
      }

      // Enough data
      alreadyRead += len;
      return len;
    }
  }
}
