/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.streaming;

import static java.lang.Thread.currentThread;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Bidirectional blocking buffer implementation, where read() methods wait for available data before returning. Both the writer
 * and the reader sides can {@link #close()} this structure, which means that no more data will/can be written. When the structure
 * is closed, the reader side won't block if there is no data available. It provides implementations of {@link InputStream} and
 * {@link OutputStream} in order to interact with this structure, via the methods {@link #getInputStream()} and
 * {@link #getOutputStream()} respectively.
 */
public class BlockingBidirectionalStream {

  private final ReentrantLock lock;

  private final BidirectionalByteBufferStream bidiBuffer;
  private boolean isClosed;
  private Throwable cancellationError;

  private final BlockingInputStreamView inputStream;
  private final BlockingOutputStreamView outputStream;
  private final Condition emptyCondition;

  public BlockingBidirectionalStream() {
    this.bidiBuffer = new BidirectionalByteBufferStream();
    this.isClosed = false;

    this.lock = new ReentrantLock();
    this.emptyCondition = lock.newCondition();

    this.inputStream = new BlockingInputStreamView(this);
    this.outputStream = new BlockingOutputStreamView(this);
  }

  /**
   * @return an implementation of {@link InputStream} that allows the consumer side to interact with this data structure with a
   *         well-known Java interface.
   */
  public InputStream getInputStream() {
    return inputStream;
  }

  /**
   * @return an implementation of {@link OutputStream} that allows the producer side to interact with this data structure with a
   *         well-known Java interface.
   */
  public CancelableOutputStream getOutputStream() {
    return outputStream;
  }

  /**
   * Retrieves the requested amount of bytes from the stream. If no data is present in the buffer, this method blocks until there
   * is something to consume, or the structure is closed. As a result of this method call, the bytes will be consumed, which means
   * that they won't be accessible anymore in a next call.
   *
   * @param b   The buffer where the (output) bytes will be stored.
   * @param off The start offset in array <code>b</code> at which the data is written.
   * @param len the maximum number of bytes to read.
   *
   * @return the total number of bytes read into the buffer, or <code>-1</code> if there is no more data and the structure was
   *         closed.
   * @throws IOException if an error happens during the read.
   *
   * @see InputStream#read(byte[], int, int)
   * @see #close()
   */
  public int read(byte[] b, int off, int len) throws IOException {
    try {
      lock.lockInterruptibly();
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new IOException(e);
    }
    try {
      while (bidiBuffer.isEmpty() && !isClosed && cancellationError == null) {
        emptyCondition.await();
      }
      if (cancellationError != null) {
        throw new IOException("Streaming canceled by writer", cancellationError);
      } else if (bidiBuffer.isEmpty()) {
        return -1;
      } else {
        return bidiBuffer.read(b, off, len);
      }
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new IOException(e);
    } finally {
      lock.unlock();
    }
  }

  /**
   * Reads the next byte in the stream, or returns -1 if there is no more data and the structure is closed. This call is blocking
   * if there is no data available and the structure is not closed. Please use {@link #read(byte[], int, int)} instead.
   *
   * @return the next byte in the stream, or -1 if the structure is closed and there is no more data available.
   *
   * @throws IOException if some error is found while reading.
   */
  public int read() throws IOException {
    byte[] aux = new byte[1];
    int result = this.read(aux, 0, 1);
    return result == -1 ? -1 : aux[0] & 0xFF;
  }

  /**
   * Writes <code>len</code> bytes from the specified byte array starting at offset <code>off</code> to this output stream.
   *
   * @param b   the data.
   * @param off the start offset in the data.
   * @param len the number of bytes to write.
   * @throws IOException if an I/O error occurs. In particular, it is thrown if the structure is closed.
   *
   * @see OutputStream#write(byte[], int, int)
   */
  public void write(byte[] b, int off, int len) throws IOException {
    try {
      lock.lockInterruptibly();
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new IOException(e);
    }
    try {
      if (cancellationError != null) {
        throw new IOException("Trying to write in a canceled stream", cancellationError);
      } else if (isClosed) {
        throw new IOException("Trying to write in a closed buffer");
      }
      bidiBuffer.write(b, off, len);
      emptyCondition.signalAll();
    } finally {
      lock.unlock();
    }
  }

  /**
   * Same as {@link #write(byte[], int, int)}, but only writes one byte. Please use {@link #write(byte[], int, int)} instead.
   *
   * @param b the byte to be written.
   *
   * @throws IOException if an I/O error occurs. In particular, it is thrown if the structure is closed.
   */
  public void write(int b) throws IOException {
    byte[] aux = {(byte) b};
    write(aux, 0, 1);
  }

  /**
   * Indicates that the structure won't be able to receive more bytes ({@link #write(byte[], int, int)} method will fail). It also
   * will unblock the consumers that are blocked while reading, and they will start returning -1 if no more data is available.
   */
  public void close() {
    lock.lock();
    try {
      isClosed = true;
      emptyCondition.signalAll();
    } finally {
      lock.unlock();
    }
  }

  /**
   * Marks the stream with an error. That error will be propagated to the reader.
   * 
   * @param error the error.
   */
  public void cancel(Throwable error) {
    lock.lock();
    try {
      isClosed = true;
      cancellationError = error;
      emptyCondition.signalAll();
    } finally {
      lock.unlock();
    }
  }

  private static class BlockingInputStreamView extends InputStream {

    private final BlockingBidirectionalStream buffer;

    public BlockingInputStreamView(BlockingBidirectionalStream buffer) {
      this.buffer = buffer;
    }

    @Override
    public int read() throws IOException {
      return buffer.read();
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
      return buffer.read(b, off, len);
    }

    @Override
    public void close() throws IOException {
      super.close();
      buffer.close();
    }
  }

  private static class BlockingOutputStreamView extends CancelableOutputStream {

    private final BlockingBidirectionalStream buffer;

    public BlockingOutputStreamView(BlockingBidirectionalStream buffer) {
      this.buffer = buffer;
    }

    @Override
    public void write(int b) throws IOException {
      buffer.write(b);
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
      buffer.write(b, off, len);
    }

    @Override
    public void close() throws IOException {
      super.close();
      buffer.close();
    }

    @Override
    public void cancel(Throwable error) {
      buffer.cancel(error);
    }
  }
}
