/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.server.sse;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Values.TEXT_EVENT_STREAM;
import static org.mule.service.http.test.common.util.HttpMessageHeaderMatcher.hasHeader;
import static org.mule.service.http.test.netty.AllureConstants.SSE;
import static org.mule.service.http.test.netty.AllureConstants.SseStory.SSE_ENDPOINT;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentCaptor.forClass;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.sse.server.SseClientConfig;
import org.mule.service.http.common.server.sse.SseResponseStarter;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import java.io.IOException;
import java.io.StringWriter;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;

@Feature(SSE)
@Story(SSE_ENDPOINT)
public class SseResponseStarterTestCase extends AbstractHttpTestCase {

  private static final String SSE_CONTENT_TYPE = TEXT_EVENT_STREAM;

  private StringWriter serializedDataSink;
  private HttpResponseReadyCallback callback;
  private SseClientConfig mockSseClientConfig;

  @Before
  public void setup() throws Exception {
    serializedDataSink = new StringWriter();
    callback = mock(HttpResponseReadyCallback.class);
    when(callback.startResponse(any(), any(), any())).thenReturn(serializedDataSink);
    mockSseClientConfig = mock(SseClientConfig.class);
    when(mockSseClientConfig.getResponseCustomizerConsumer()).thenReturn(ignored -> {
    });
  }

  @Test
  public void startsResponseWithSseStatusCodeAndContentType() throws IOException {
    var responseCaptor = forClass(HttpResponse.class);
    var sseResponseStarter = new SseResponseStarter();

    try (var ignoredClient = sseResponseStarter.startResponse(mockSseClientConfig, callback)) {
      verify(callback).startResponse(responseCaptor.capture(), any(), any());

      var response = responseCaptor.getValue();
      assertThat(response.getStatusCode(), is(200));
      assertThat(response, hasHeader(CONTENT_TYPE, SSE_CONTENT_TYPE));
    }
  }

  @Test
  public void basicSseSend() throws IOException {
    var sseResponseStarter = new SseResponseStarter();
    try (var sseClient = sseResponseStarter.startResponse(mockSseClientConfig, callback)) {
      sseClient.sendEvent("Incredibly easy to use!");
    }

    assertThat(serializedDataSink.toString(), is("""
        event: message
        data: Incredibly easy to use!

        """));
  }
}
