/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth;

import static org.mule.runtime.http.api.HttpHeaders.Names.AUTHORIZATION;
import static org.mule.runtime.http.api.HttpHeaders.Names.WWW_AUTHENTICATE;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.client.auth.HttpAuthentication;
import org.mule.runtime.http.api.client.auth.HttpAuthentication.HttpNtlmAuthentication;
import org.mule.runtime.http.api.client.auth.HttpAuthenticationType;
import org.mule.service.http.netty.impl.client.auth.ntlm.NtlmMessageFactory;

import java.net.URI;

import io.netty.handler.codec.http.DefaultHttpHeaders;
import io.netty.handler.codec.http.HttpHeaders;
import org.slf4j.Logger;

public class AuthenticationEngine {

  private static final Logger LOGGER = getLogger(AuthenticationEngine.class);

  private final HttpAuthenticationType type;
  private final AuthHeaderFactory authMessageFactory;

  public AuthenticationEngine(HttpAuthentication authentication, URI uri, String method) {
    this.type = authentication.getType();
    this.authMessageFactory = createAuthHeaderFactory(authentication, uri, method);
  }

  private AuthHeaderFactory createAuthHeaderFactory(HttpAuthentication authentication, URI uri, String method) {
    switch (type) {
      case BASIC:
        return new BasicAuthHeaderFactory(authentication.isPreemptive(), authentication.getUsername(),
                                          authentication.getPassword());
      case NTLM:
        HttpNtlmAuthentication ntlmAuthentication = (HttpNtlmAuthentication) authentication;
        return new NtlmMessageFactory(ntlmAuthentication.getDomain(), ntlmAuthentication.getWorkstation(),
                                      authentication.getUsername(), authentication.getPassword());
      case DIGEST:
        return new DigestAuthMessageFactory(authentication.getUsername(), authentication.getPassword(), uri, method);
      default:
        throw new IllegalArgumentException("Unknown authentication type: " + type);
    }
  }

  /**
   * Convenience method to generate an "Authorization" header value according to the different status the client is in. In some
   * cases it might use an "WWW-Authenticate" header.
   *
   * @param wwwAuthenticateHeader value of the received "WWW-Authenticate" header.
   * @return value to be sent in the "Authorization" header of the next request. Can be {@code null}, and it means that the header
   *         has not to be included.
   * @throws Exception Some message computations might return exceptions
   */
  public String getNextHeader(String wwwAuthenticateHeader) throws Exception {
    return authMessageFactory.getNextHeader(wwwAuthenticateHeader);
  }

  public HttpHeaders getAuthHeaders(HttpHeaders currentHeaders) {
    HttpHeaders authHeaders = new DefaultHttpHeaders();
    String authHeader = computeAuthHeaders(currentHeaders.get(WWW_AUTHENTICATE));
    if (authHeader != null) {
      authHeaders.add(AUTHORIZATION, authHeader);
    }
    return authHeaders;
  }

  private String computeAuthHeaders(String wwwAuthenticateHeader) {
    try {
      return getNextHeader(wwwAuthenticateHeader);
    } catch (Exception e) {
      LOGGER.warn("Error while calculating {} Auth header", type, e);
      return null;
    }
  }

  public HttpAuthenticationType getType() {
    return this.type;
  }

  public boolean hasFinished() {
    return authMessageFactory.hasFinished();
  }
}
