/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.proxy;

import static io.netty.buffer.Unpooled.EMPTY_BUFFER;
import static io.netty.handler.codec.http.DefaultHttpHeadersFactory.headersFactory;
import static io.netty.handler.codec.http.DefaultHttpHeadersFactory.trailersFactory;
import static io.netty.handler.codec.http.HttpHeaderNames.PROXY_AUTHENTICATE;
import static io.netty.handler.codec.http.HttpHeaderNames.PROXY_AUTHORIZATION;
import static io.netty.handler.codec.http.HttpMethod.CONNECT;
import static io.netty.handler.codec.http.HttpUtil.formatHostnameForHttp;
import static io.netty.handler.codec.http.HttpVersion.HTTP_1_1;

import org.mule.runtime.http.api.client.proxy.ProxyConfig;

import java.net.InetSocketAddress;
import java.net.SocketAddress;

import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpClientCodec;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpHeadersFactory;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.HttpResponse;
import io.netty.handler.codec.http.HttpResponseStatus;
import io.netty.handler.codec.http.LastHttpContent;
import io.netty.handler.proxy.HttpProxyHandler.HttpProxyConnectException;
import io.netty.handler.proxy.ProxyHandler;

/**
 * Handler that establishes a blind forwarding proxy tunnel using
 * <a href="https://datatracker.ietf.org/doc/html/rfc7231#section-4.3.6">HTTP/1.1 CONNECT</a> request. It can be used to establish
 * plaintext or secure tunnels.
 * <p>
 * It's based on Netty's {@link io.netty.handler.proxy.HttpProxyHandler}, but it adds the {@link ProxyAuthenticator} to decouple
 * the authentication headers handling.
 * <p>
 * HTTP users who need to connect to a <a href="https://datatracker.ietf.org/doc/html/rfc7230#page-10">message-forwarding HTTP
 * proxy agent</a> instead of a tunneling proxy should not use this handler, but {@link MessageForwardingProxyClientHandler}.
 */
public class BlindTunnelingProxyClientHandler extends ProxyHandler {

  private static final String PROTOCOL = "http";

  // This handler will encode/decode only the CONNECT methods to the proxy server. The requests to the target server will
  // be encoded by the other codecs in the pipeline.
  private final HttpClientCodec connectCodec = new HttpClientCodec();

  private final ProxyAuthenticator proxyAuthenticator;

  // As we receive the response in parts (not aggregated), we may need to save the status and headers between the header
  // part of the request and the last content (see handleResponse).
  private HttpResponseStatus status;
  private HttpHeaders inboundHeaders;

  public BlindTunnelingProxyClientHandler(SocketAddress proxyAddress, ProxyAuthenticator proxyAuthenticator) {
    super(proxyAddress);
    this.proxyAuthenticator = proxyAuthenticator;
  }

  public BlindTunnelingProxyClientHandler(ProxyConfig proxyConfig) {
    this(new InetSocketAddress(proxyConfig.getHost(), proxyConfig.getPort()), new ProxyAuthenticator(proxyConfig));
  }

  @Override
  public String protocol() {
    return PROTOCOL;
  }

  @Override
  public String authScheme() {
    return proxyAuthenticator.getAuthScheme();
  }

  @Override
  protected void addCodec(ChannelHandlerContext ctx) throws Exception {
    String name = ctx.name();
    ctx.pipeline().addBefore(name, "ConnectMethodsCodec", connectCodec);
  }

  @Override
  protected void removeEncoder(ChannelHandlerContext ctx) throws Exception {
    connectCodec.removeOutboundHandler();
  }

  @Override
  protected void removeDecoder(ChannelHandlerContext ctx) throws Exception {
    connectCodec.removeInboundHandler();
  }

  @Override
  protected Object newInitialMessage(ChannelHandlerContext ctx) throws Exception {
    return newConnectRequest(null);
  }

  private HttpRequest newConnectRequest(String receivedAuthenticateHeader) throws Exception {
    InetSocketAddress targetAddress = destinationAddress();

    String hostString = formatHostnameForHttp(targetAddress);
    int port = targetAddress.getPort();
    String url = hostString + ":" + port;

    HttpHeadersFactory headersFactory = headersFactory().withValidation(false);
    HttpHeadersFactory trailersFactory = trailersFactory().withValidation(false);
    FullHttpRequest request = new DefaultFullHttpRequest(HTTP_1_1, CONNECT,
                                                         url,
                                                         EMPTY_BUFFER, headersFactory, trailersFactory);

    request.headers().set(HttpHeaderNames.HOST, url);

    if (!proxyAuthenticator.hasFinished()) {
      String firstAuthHeader = proxyAuthenticator.getNextHeader(receivedAuthenticateHeader);
      if (firstAuthHeader != null) {
        request.headers().set(PROXY_AUTHORIZATION, firstAuthHeader);
      }
    }

    return request;
  }

  @Override
  protected boolean handleResponse(ChannelHandlerContext ctx, Object obj) throws Exception {
    if (obj instanceof HttpResponse response) {
      // Handle the header. If we have to send something to the server, it will happen when we receive the LastHttpContent below.
      if (status != null) {
        // This is an illegal state...
        throw new HttpProxyConnectException(exceptionMessage("too many responses"), /* headers= */ null);
      }
      status = response.status();
      inboundHeaders = response.headers();
    }

    if (obj instanceof LastHttpContent) {
      if (status == null || inboundHeaders == null) {
        // This is an illegal state...
        throw new HttpProxyConnectException(exceptionMessage("missing response"), inboundHeaders);
      }

      if (status.code() == 200) {
        // The proxy authentication is successful, proceed.
        return true;
      }

      if (!proxyAuthenticator.hasFinished()) {
        HttpRequest connectRequest = newConnectRequest(inboundHeaders.get(PROXY_AUTHENTICATE));
        sendToProxyServer(connectRequest);
        status = null;
        inboundHeaders = null;

        // We sent an auth connect, the authentication dance didn't finish.
        return false;
      }

      // The code is not a 200, and the auth finished, so it's a definitive error.
      throw new HttpProxyConnectException(exceptionMessage("status: " + status), inboundHeaders);
    }

    return false;
  }
}

