/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client.sse;

import static org.mule.service.http.test.netty.AllureConstants.SSE;
import static org.mule.service.http.test.netty.AllureConstants.SseStory.SSE_SOURCE;
import static org.mule.service.http.test.netty.utils.sse.ServerSentEventTypeSafeMatcher.aServerSentEvent;

import static java.lang.Math.min;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.is;
import static org.mockito.ArgumentCaptor.forClass;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;

import org.mule.runtime.http.api.sse.ServerSentEvent;
import org.mule.runtime.http.api.sse.client.SseListener;
import org.mule.service.http.common.client.sse.ServerSentEventDecoder;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import io.qameta.allure.Feature;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;

@Feature(SSE)
@Story(SSE_SOURCE)
public class ServerSentEventDecoderTestCase extends AbstractHttpTestCase {

  private SseListener eventListener;
  private ServerSentEventDecoder decoder;

  @Before
  public void setUp() {
    eventListener = mock(SseListener.class);
    decoder = new ServerSentEventDecoder(eventListener);
  }

  @Test
  public void happyPathDecode() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedDecoder(decoder, """
        event: message
        data: message1; line1
        data: message1; line2
        id: message1

        event: message
        data: message2; line1
        data: message2; line2
        id: message2
        retry: 1000

        """);

    verify(eventListener, times(2)).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();
    assertThat(eventsCaptor.getAllValues(), contains(
                                                     aServerSentEvent("message", "message1; line1\nmessage1; line2", "message1",
                                                                      null),
                                                     aServerSentEvent("message", "message2; line1\nmessage2; line2", "message2",
                                                                      1000L)));
  }

  @Test
  public void dataWithEmptyLines() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedDecoder(decoder, """
        event: message
        data: message1; line1
        data:
        data: message1; line3

        """);

    verify(eventListener).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();

    ServerSentEvent event = eventsCaptor.getValue();
    assertThat(event, is(aServerSentEvent("message", "message1; line1\n\nmessage1; line3")));
  }

  @Test
  public void retryIsNotANumber() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedDecoder(decoder, """
        event: message
        data: data
        retry: thisIsAString

        """);

    verify(eventListener).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();

    ServerSentEvent eventParsed = eventsCaptor.getValue();
    assertThat(eventParsed, is(aServerSentEvent("message", "data")));
    assertThat("Retry should be ignored because it is not a number", eventParsed.getRetryDelay().isPresent(), is(false));
  }

  @Test
  public void unexpectedEventFieldIsJustIgnored() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedDecoder(decoder, """
        event: message
        data: data
        thisIsNotExpected: irrelevantValue

        """);

    verify(eventListener).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();

    ServerSentEvent eventParsed = eventsCaptor.getValue();
    assertThat(eventParsed, is(aServerSentEvent("message", "data")));
  }


  @Test
  @Issue("W-19667148")
  public void eventWithoutEventLineDefaultsToMessage() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedDecoder(decoder, """
        data: data

        """);

    verify(eventListener).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();

    ServerSentEvent eventParsed = eventsCaptor.getValue();
    assertThat(eventParsed, is(aServerSentEvent("message", "data")));
  }

  private static void feedDecoder(ServerSentEventDecoder sseDecoder, String payload) {
    try (sseDecoder) {
      int chunkLength = 10;
      int begin = 0;
      int length = payload.length();
      while (begin < length) {
        int end = min(begin + chunkLength, length);
        String chunk = payload.substring(begin, end);
        sseDecoder.decode(chunk.getBytes(), chunk.length());
        begin += chunkLength;
      }
    }
  }
}
