/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth.ntlm.av;

import static java.nio.charset.StandardCharsets.UTF_16LE;

import java.nio.charset.Charset;

/**
 * Represents the AV_PAIR structure for a target name in NTLM authentication. This class encodes and decodes the target name using
 * UTF-16LE encoding.
 *
 * <p>
 * This implementation is based on the jcifs library, available at:
 * <a href="https://github.com/codelibs/jcifs">https://github.com/codelibs/jcifs</a>
 * </p>
 *
 * @see <a href="https://github.com/codelibs/jcifs">jcifs on GitHub</a>
 */
public class AvTargetName extends AvPair {

  private static final Charset UTF16LE = UTF_16LE;

  /**
   * Constructs an instance of {@code AvTargetName} with the specified raw byte data.
   *
   * @param raw the raw byte data representing the target name.
   */
  public AvTargetName(byte[] raw) {
    super(MsvAvTargetName, raw);
  }

  /**
   * Constructs an instance of {@code AvTargetName} with the specified target name string.
   *
   * @param targetName the target name to encode.
   */
  public AvTargetName(String targetName) {
    this(encode(targetName));
  }

  /**
   * Encodes a target name string into a UTF-16LE byte array.
   *
   * @param targetName the target name to encode.
   * @return a byte array containing the UTF-16LE encoded target name.
   */
  private static byte[] encode(String targetName) {
    return targetName.getBytes(UTF16LE);
  }
}
