/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client.sse;

import static org.mule.service.http.test.netty.AllureConstants.SSE;
import static org.mule.service.http.test.netty.AllureConstants.SseStory.SSE_SOURCE;
import static org.mule.service.http.test.netty.utils.sse.ServerSentEventTypeSafeMatcher.aServerSentEvent;

import static java.nio.charset.StandardCharsets.UTF_8;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThrows;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;
import static org.mockito.ArgumentCaptor.forClass;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.http.api.sse.ServerSentEvent;
import org.mule.runtime.http.api.sse.client.SseListener;
import org.mule.service.http.common.client.sse.ServerSentEventDataListener;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;

@Feature(SSE)
@Story(SSE_SOURCE)
public class ServerSentEventDataListenerTestCase extends AbstractHttpTestCase {

  private SseListener eventListener;
  private ServerSentEventDataListener dataListener;

  @Before
  public void setUp() {
    eventListener = mock(SseListener.class);
    dataListener = new ServerSentEventDataListener(eventListener, "TestClient");
  }

  @Test
  public void happyPathDecode() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedListener(dataListener, """
        event: message
        data: message1; line1
        data: message1; line2
        id: message1

        event: message
        data: message2; line1
        data: message2; line2
        id: message2
        retry: 1000

        """);

    verify(eventListener, times(2)).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();
    assertThat(eventsCaptor.getAllValues(), contains(
                                                     aServerSentEvent("message", "message1; line1\nmessage1; line2", "message1",
                                                                      null),
                                                     aServerSentEvent("message", "message2; line1\nmessage2; line2", "message2",
                                                                      1000L)));
  }

  @Test
  public void dataWithEmptyLines() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedListener(dataListener, """
        event: message
        data: message1; line1
        data:
        data: message1; line3

        """);

    verify(eventListener).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();

    ServerSentEvent event = eventsCaptor.getValue();
    assertThat(event, is(aServerSentEvent("message", "message1; line1\n\nmessage1; line3")));
  }

  @Test
  public void retryIsNotANumber() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedListener(dataListener, """
        event: message
        data: data
        retry: thisIsAString

        """);

    verify(eventListener).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();

    ServerSentEvent eventParsed = eventsCaptor.getValue();
    assertThat(eventParsed, is(aServerSentEvent("message", "data")));
    assertThat("Retry should be ignored because it is not a number", eventParsed.getRetryDelay().isPresent(), is(false));
  }

  @Test
  public void unexpectedEventFieldIsJustIgnored() {
    ArgumentCaptor<ServerSentEvent> eventsCaptor = forClass(ServerSentEvent.class);
    feedListener(dataListener, """
        event: message
        data: data
        thisIsNotExpected: irrelevantValue

        """);

    verify(eventListener).onEvent(eventsCaptor.capture());
    verify(eventListener).onClose();

    ServerSentEvent eventParsed = eventsCaptor.getValue();
    assertThat(eventParsed, is(aServerSentEvent("message", "data")));
  }

  @Test
  public void streamHasLessBytesThanExpectedSoItReadsWhatItHas() throws IOException {
    String payload = "payload";
    InputStream stream = new ByteArrayInputStream(payload.getBytes(UTF_8));
    assertThat(stream.available(), is(payload.length()));
    dataListener.onStreamCreated(stream);
    dataListener.onDataAvailable(payload.length() + 1);
    assertThat(stream.available(), is(0));
  }

  @Test
  public void streamThrowsIOException() throws IOException {
    InputStream stream = mock(InputStream.class);
    when(stream.read(any())).thenThrow(IOException.class);
    dataListener.onStreamCreated(stream);
    var exception = assertThrows(IllegalStateException.class, () -> dataListener.onDataAvailable(5));
    assertThat(exception, hasMessage(containsString("Exception parsing events stream")));
  }

  private static void feedListener(ServerSentEventDataListener sseDataListener, String payload) {
    InputStream inputStream = new ByteArrayInputStream(payload.getBytes(UTF_8));
    int length = payload.length();

    int chunkLength = 10;
    int numberOfCompleteChunks = length / chunkLength;
    int extraChunkLength = length % chunkLength;
    sseDataListener.onStreamCreated(inputStream);
    for (int i = 0; i < numberOfCompleteChunks; i++) {
      sseDataListener.onDataAvailable(chunkLength);
    }
    sseDataListener.onDataAvailable(extraChunkLength);
    sseDataListener.onEndOfStream();
  }
}
