/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static org.mule.service.http.test.netty.AllureConstants.HttpStory.TIMEOUTS;
import static org.mule.service.http.test.netty.utils.TestUtils.measuringNanoseconds;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

import static org.apache.commons.lang3.SystemUtils.IS_OS_MAC;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsStringIgnoringCase;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.lessThan;
import static org.junit.Assert.assertThrows;
import static org.junit.Assume.assumeThat;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;

import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.tcp.TcpClientSocketProperties;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;

import io.qameta.allure.Story;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

@Story(TIMEOUTS)
public class ClientTcpConnectionTimeoutTestCase extends AbstractHttpServiceTestCase {

  private static final Integer CLIENT_CONNECTION_TIMEOUT_MILLIS = 1;

  @Rule
  public DynamicPort tcpServerPort = new DynamicPort("tcpServerPort");

  private HttpClient client;

  public ClientTcpConnectionTimeoutTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Before
  public void setup() {
    var withConnectionTimeout = TcpClientSocketProperties.builder()
        .connectionTimeout(CLIENT_CONNECTION_TIMEOUT_MILLIS)
        .build();
    var clientConfig = new HttpClientConfiguration.Builder()
        .setName("client-with-connection-timeout")
        .setClientSocketProperties(withConnectionTimeout)
        .build();
    client = service.getClientFactory().create(clientConfig);
    client.start();
  }

  @After
  public void tearDown() {
    if (client != null) {
      client.stop();
    }
  }

  @Test
  public void clientTcpConnectionTimeout() throws Exception {
    // Backlog configuration is just a hint for Linux kernel, so I can't be sure that a connection is
    // in the backlog queue.
    // If you can, please adapt the test.
    assumeThat(IS_OS_MAC, is(true));

    // When using Java's ServerSocket, we have two different listen backlog queues:
    // 1 - The tcp "syn" queue is the operative system one...
    // 2 - Java accepts the connections and move them to an auxiliary queue, the "accept" queue, and the accept() method will just
    // get a connection from THAT queue.

    // The serverSocket will be bound and listening within the try, but we won't call accept()
    try (var serverSocket = new ServerSocket(tcpServerPort.getNumber(), 1)) {
      // Sending a client socket to fill the "accept" queue (the "syn" queue will remain empty).
      try (var auxClientSocket = new Socket("localhost", tcpServerPort.getNumber())) {
        // Then, our client connection will go to the syn queue of the server, and nobody will accept it
        var request = HttpRequest.builder()
            .uri("http://localhost:" + tcpServerPort.getNumber())
            .build();

        long elapsedNanos = measuringNanoseconds(() -> {
          var error = assertThrows(IOException.class, () -> client.send(request));
          assertThat(error, hasMessage(containsStringIgnoringCase("timeout")));
        });
        long expectedTimeoutNanos = MILLISECONDS.toNanos(CLIENT_CONNECTION_TIMEOUT_MILLIS);
        long toleranceNanos = SECONDS.toNanos(1); // default connection timeout is 30 seconds...
        assertThat(elapsedNanos, lessThan(expectedTimeoutNanos + toleranceNanos));
      }
    }
  }
}
