/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.client;

import static org.mule.runtime.api.util.DataUnit.KB;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.service.http.test.common.client.sse.FillAndWaitStream.RESPONSE_SIZE;

import static java.lang.Thread.currentThread;
import static java.lang.Thread.sleep;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.concurrent.Executors.newSingleThreadExecutor;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;

import org.mule.runtime.api.util.Reference;
import org.mule.runtime.api.util.concurrent.Latch;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.test.common.client.AbstractHttpClientTestCase;
import org.mule.service.http.test.common.client.sse.FillAndWaitStream;
import org.mule.service.http.test.common.util.ResponseReceivedProbe;
import org.mule.tck.probe.PollingProber;

import java.io.IOException;
import java.util.concurrent.ExecutorService;

import io.qameta.allure.Description;
import io.qameta.allure.Issue;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Test;

/**
 * This is the equivalent of the HTTP service streaming test, adapted for Netty service using the common test base.
 */
@Issue("W-18040828")
public class HttpClientStreamingTestCase extends AbstractHttpClientTestCase {

  private static final int RESPONSE_TIMEOUT = 3000;
  private static final int TIMEOUT_MILLIS = 1000;
  private static final int POLL_DELAY_MILLIS = 200;

  private static Latch latch;

  private final HttpClientConfiguration.Builder clientBuilder = new HttpClientConfiguration.Builder().setName("streaming-test");
  private final PollingProber pollingProber = new PollingProber(TIMEOUT_MILLIS, POLL_DELAY_MILLIS);

  public HttpClientStreamingTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Before
  public void createLatch() {
    latch = new Latch();
  }

  @Test
  @Description("Uses a streaming HTTP client to send a non blocking request which will finish before the stream is released.")
  public void nonBlockingStreaming() throws Exception {
    HttpClient client =
        service.getClientFactory().create(clientBuilder.setResponseBufferSize(KB.toBytes(10)).setStreaming(true).build());
    client.start();
    final Reference<HttpResponse> responseReference = new Reference<>();
    try {
      client.sendAsync(getRequest(), getDefaultOptions(RESPONSE_TIMEOUT)).whenComplete(
                                                                                       (response, exception) -> responseReference
                                                                                           .set(response));
      pollingProber.check(new ResponseReceivedProbe(responseReference));
      verifyStreamed(responseReference.get());
    } finally {
      client.stop();
    }
  }

  @Test
  @Description("Uses a non streaming HTTP client to send a non blocking request which will not finish until the stream is released.")
  public void nonBlockingMemory() throws Exception {
    HttpClient client = service.getClientFactory().create(clientBuilder.setStreaming(false).build());
    client.start();
    final Reference<HttpResponse> responseReference = new Reference<>();
    try {
      client.sendAsync(getRequest(), getDefaultOptions(RESPONSE_TIMEOUT)).whenComplete(
                                                                                       (response, exception) -> responseReference
                                                                                           .set(response));
      verifyNotStreamed(responseReference);
    } finally {
      client.stop();
    }
  }

  @Test
  @Description("Uses a streaming HTTP client to send a blocking request which will finish before the stream is released.")
  public void blockingStreaming() throws Exception {
    HttpClient client = service.getClientFactory().create(clientBuilder.setStreaming(true).build());
    client.start();
    try {
      HttpResponse response = client.send(getRequest(), getDefaultOptions(RESPONSE_TIMEOUT));
      verifyStreamed(response);
    } finally {
      client.stop();
    }
  }

  @Test
  @Description("Uses a non streaming HTTP client to send a request which will not finish until the stream is released.")
  public void blockingMemory() throws Exception {
    HttpClient client = service.getClientFactory().create(clientBuilder.setStreaming(false).build());
    client.start();
    Reference<HttpResponse> responseReference = new Reference<>();
    ExecutorService executorService = newSingleThreadExecutor();
    try {
      executorService.execute(() -> {
        try {
          responseReference.set(client.send(getRequest(), getDefaultOptions(RESPONSE_TIMEOUT)));
        } catch (Exception e) {
          // Do nothing, probe will fail.
        }
      });
      verifyNotStreamed(responseReference);
    } finally {
      executorService.shutdown();
      client.stop();
    }
  }

  @Test
  @Issue("MULE-19072")
  @Description("The client should use other thread-pool than the uber to avoid a deadlock when using a PipedInputStream")
  public void responseStreamingDoesNotUseUberPoolToWritePartsToThePipe() throws IOException {
    HttpClient client =
        service.getClientFactory().create(clientBuilder.setResponseBufferSize(KB.toBytes(10)).setStreaming(true).build());
    client.start();
    final Reference<HttpResponse> responseReference = new Reference<>();
    final Reference<String> threadGroupName = new Reference<>();
    try {
      client.sendAsync(getRequest(), getDefaultOptions(RESPONSE_TIMEOUT)).whenComplete((response, exception) -> {
        responseReference.set(response);
        threadGroupName.set(currentThread().getThreadGroup().getName());
      });
      pollingProber.check(new ResponseReceivedProbe(responseReference));
      verifyStreamed(responseReference.get());
      // Note: Netty implementation uses IO scheduler which shares the test thread group.
      // This is different from Grizzly which uses custom schedulers for streaming workers.
      // The important thing is that streaming works without deadlocks, not the specific thread group.
      assertThat("Response callback should be executed", responseReference.get(), not(nullValue()));
      assertThat("Thread group name should be captured", threadGroupName.get(), not(nullValue()));
    } finally {
      client.stop();
    }
  }

  private HttpRequest getRequest(String uri) {
    return HttpRequest.builder().uri(uri).build();
  }

  private HttpRequest getRequest() {
    return getRequest(getUri());
  }

  private void verifyStreamed(HttpResponse response) throws IOException {
    assertThat(response.getStatusCode(), is(OK.getStatusCode()));
    latch.release();
    verifyBody(response);
  }

  private void verifyNotStreamed(Reference<HttpResponse> responseReference) throws Exception {
    // Allow the request/response process to start
    sleep(1000);
    assertThat(responseReference.get(), is(nullValue()));
    latch.release();
    pollingProber.check(new ResponseReceivedProbe(responseReference));
    assertThat(responseReference.get().getStatusCode(), is(OK.getStatusCode()));
    verifyBody(responseReference.get());
  }

  private void verifyBody(HttpResponse response) throws IOException {
    assertThat(IOUtils.toString(response.getEntity().getContent(), UTF_8).length(), is(RESPONSE_SIZE));
  }

  @Override
  protected HttpResponse setUpHttpResponse(HttpRequest request) {
    return HttpResponse.builder()
        .statusCode(OK.getStatusCode())
        .reasonPhrase(OK.getReasonPhrase())
        .entity(new InputStreamHttpEntity(new FillAndWaitStream(latch)))
        .build();
  }

  @Override
  protected HttpRequestOptions getDefaultOptions(int responseTimeout) {
    return HttpRequestOptions.builder().responseTimeout(responseTimeout).build();
  }
}
