/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.common.message.sse;

import org.mule.runtime.http.api.sse.ServerSentEvent;

/**
 * Builds instances of {@link ServerSentEvent}.
 */
public class SseEventBuilder {

  // See https://html.spec.whatwg.org/multipage/server-sent-events.html#event-stream-interpretation
  private static final String DEFAULT_NAME = "message";

  private String eventName;
  private String id;
  private final StringBuilder eventData = new StringBuilder();
  private Long retryDelay;

  public SseEventBuilder() {
    reset();
  }

  public ServerSentEvent buildAndClear() {
    ServerSentEvent event = new ServerSentEventImpl(eventName, eventData.toString(), id, retryDelay);
    reset();
    return event;
  }

  public SseEventBuilder withData(String dataLine) {
    if (!eventData.isEmpty()) {
      eventData.append("\n");
    }
    eventData.append(dataLine);
    return this;
  }

  public SseEventBuilder withName(String eventName) {
    this.eventName = eventName;
    return this;
  }

  public SseEventBuilder withId(String id) {
    this.id = id;
    return this;
  }

  public SseEventBuilder withRetryDelay(Long retryDelay) {
    this.retryDelay = retryDelay;
    return this;
  }

  private void reset() {
    eventName = DEFAULT_NAME;
    id = null;
    eventData.setLength(0);
    retryDelay = null;
  }
}
