/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.service.http.netty.impl.server.util.HttpParser.fromMultipartEntity;
import static org.mule.service.http.netty.impl.util.MuleToNettyUtils.adaptResponseWithoutBody;

import static java.util.Collections.emptyMap;
import static java.util.UUID.randomUUID;

import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.multipart.MultipartHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.runtime.http.api.sse.server.SseClient;
import org.mule.runtime.http.api.sse.server.SseClientConfig;
import org.mule.service.http.common.server.sse.SseResponseStarter;
import org.mule.service.http.netty.impl.server.sse.ResponseBodyWriter;

import java.io.IOException;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.concurrent.Executor;

import io.netty.channel.ChannelHandlerContext;

public class NettyHttp1ResponseReadyCallback implements HttpResponseReadyCallback {

  private final ChannelHandlerContext ctx;
  private final HttpRequest httpRequest;
  private final Executor ioExecutor;

  public NettyHttp1ResponseReadyCallback(ChannelHandlerContext ctx, HttpRequest muleRequest, Executor ioExecutor) {
    this.ctx = ctx;
    this.httpRequest = muleRequest;
    this.ioExecutor = ioExecutor;
  }

  @Override
  public void responseReady(HttpResponse response, ResponseStatusCallback responseStatusCallback) {
    try {
      sendResponse(ctx, response, responseStatusCallback);
    } catch (IOException e) {
      responseStatusCallback.responseSendFailure(e);
    }
  }

  @Override
  public Writer startResponse(HttpResponse response, ResponseStatusCallback responseStatusCallback, Charset encoding) {
    var nettyResponse = adaptResponseWithoutBody(response, httpRequest.getProtocol());
    ctx.writeAndFlush(nettyResponse);
    return new ResponseBodyWriter(ctx, encoding, responseStatusCallback);
  }

  @Override
  public SseClient startSseResponse(SseClientConfig config) {
    return new SseResponseStarter().startResponse(config, this);
  }

  private void sendResponse(ChannelHandlerContext ctx, HttpResponse response, ResponseStatusCallback callback)
      throws IOException {
    HttpResponse httpResponse = response;
    boolean isHeadMethod = isHeadMethod(httpRequest);
    if (isHeadMethod) {
      // Given the RFC-7231, section 4.3.2:
      // The HEAD method is identical to GET except that the server MUST NOT
      // send a message body in the response (i.e., the response terminates at
      // the end of the header section).
      // For reference: https://datatracker.ietf.org/doc/html/rfc7231#section-4.3.2
      httpResponse = buildResponseWithEmptyBody(httpResponse);
    } else if (httpResponse.getEntity().isComposed()) {
      httpResponse = buildMultipartResponse(httpResponse);
    }

    if (httpResponse.getEntity().isStreaming()) {
      new StreamingResponseSender(httpRequest, ctx, httpResponse, callback, ioExecutor).send();
    } else {
      new DirectResponseSender(httpRequest, ctx, httpResponse, callback).send();
    }

    if (isHeadMethod) {
      ctx.close();
    }
  }

  private static HttpResponse buildResponseWithEmptyBody(HttpResponse response) throws IOException {
    if (response.getEntity().isStreaming()) {
      response.getEntity().getContent().close();
    }
    return new HttpResponseBuilder(response).entity(new EmptyHttpEntity()).build();
  }

  private boolean isHeadMethod(HttpRequest httpRequest) {
    return "HEAD".equals(httpRequest.getMethod());
  }

  private static HttpResponse buildMultipartResponse(HttpResponse response) throws IOException {
    final HttpResponseBuilder httpResponseBuilder = new HttpResponseBuilder(response);

    return httpResponseBuilder
        .entity(fromMultipartEntity(response.getHeaderValue(CONTENT_TYPE),
                                    (MultipartHttpEntity) response.getEntity(),
                                    ct -> {
                                      httpResponseBuilder.removeHeader(CONTENT_TYPE);
                                      httpResponseBuilder.addHeader(CONTENT_TYPE, ct);
                                    },
                                    emptyMap()))
        .build();
  }

  private static String generateBoundary() {
    return "----MuleMultipart" + randomUUID().toString().replace("-", "");
  }

}
