/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message;

import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;

import static java.lang.Integer.getInteger;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils;

import java.io.InputStream;

import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpResponse;
import org.slf4j.Logger;
import reactor.netty.http.client.HttpClientResponse;

public class HttpResponseCreator {

  private static final Logger LOGGER = getLogger(HttpResponseCreator.class);
  private static final String HEADER_CONTENT_TYPE = CONTENT_TYPE.toLowerCase();
  private static final String HEADER_CONTENT_LENGTH = CONTENT_LENGTH.toLowerCase();
  private static final int MAX_NUM_HEADERS_DEFAULT = 100;
  private static final String MAX_SERVER_RESPONSE_HEADERS_KEY = SYSTEM_PROPERTY_PREFIX + "http.MAX_SERVER_RESPONSE_HEADERS";
  private static int MAX_SERVER_RESPONSE_HEADERS = getInteger(MAX_SERVER_RESPONSE_HEADERS_KEY, MAX_NUM_HEADERS_DEFAULT);
  private final HttpResponseCreatorUtils httpResponseCreatorUtils;

  public HttpResponseCreator() {
    this.httpResponseCreatorUtils = new HttpResponseCreatorUtils();
  }

  private static boolean checkMaxServerResponseHeadersLimit(HttpHeaders headers) {
    if (headers.size() > MAX_SERVER_RESPONSE_HEADERS) {
      LOGGER.warn("Exceeded max server response headers limit: {}. Current header count (including default headers): {}",
                  MAX_SERVER_RESPONSE_HEADERS, headers.size());
      return false;
    }
    return true;
  }

  public static void refreshMaxServerResponseHeaders() {
    MAX_SERVER_RESPONSE_HEADERS = getInteger(MAX_SERVER_RESPONSE_HEADERS_KEY, MAX_NUM_HEADERS_DEFAULT);
  }

  public static int getMaxServerResponseHeaders() {
    return MAX_SERVER_RESPONSE_HEADERS;
  }

  public org.mule.runtime.http.api.domain.message.response.HttpResponse create(HttpResponse response, InputStream inputStream) {

    HttpResponseBuilder responseBuilder = httpResponseCreatorUtils.buildResponseHeaders(response);
    String contentType = httpResponseCreatorUtils.extractHeader(response, HEADER_CONTENT_TYPE);
    String contentLength = httpResponseCreatorUtils.extractHeader(response, HEADER_CONTENT_LENGTH);

    responseBuilder.entity(
                           httpResponseCreatorUtils.createEntity(inputStream, contentType, contentLength,
                                                                 response.status().code()));

    return responseBuilder.build();
  }

  public org.mule.runtime.http.api.domain.message.response.HttpResponse create(HttpClientResponse response,
                                                                               InputStream inputStream) {
    // TODO decide if we want to return a handler that throws 500 or another appropriate response
    if (!checkMaxServerResponseHeadersLimit(response.responseHeaders())) {
      throw new IllegalArgumentException("Exceeded max server response headers limit: " + MAX_SERVER_RESPONSE_HEADERS);
    }

    HttpResponseBuilder responseBuilder = httpResponseCreatorUtils.buildResponseHeaders(response);

    String contentType = httpResponseCreatorUtils.extractHeader(response, HEADER_CONTENT_TYPE);
    String contentLength = httpResponseCreatorUtils.extractHeader(response, HEADER_CONTENT_LENGTH);

    responseBuilder.entity(
                           httpResponseCreatorUtils.createEntity(inputStream, contentType, contentLength,
                                                                 response.status().code()));

    return responseBuilder.build();
  }
}
