/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.streaming;

import static java.lang.Math.min;
import static java.lang.Math.toIntExact;
import static java.lang.System.nanoTime;
import static java.util.concurrent.TimeUnit.NANOSECONDS;

import static io.netty.handler.codec.http.LastHttpContent.EMPTY_LAST_CONTENT;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.service.http.netty.impl.server.FinishStreamingListener;
import org.mule.service.http.netty.impl.server.SendNextChunkListener;

import java.io.IOException;
import java.io.InputStream;
import java.util.OptionalLong;
import java.util.concurrent.Executor;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http.DefaultHttpContent;
import io.netty.handler.codec.http.DefaultLastHttpContent;
import io.netty.handler.codec.http.HttpContent;

/**
 * Writes a {@link HttpEntity} to a {@link ChannelHandlerContext} in a buffered fashion.
 */
public class StreamingEntitySender {

  private static final Logger LOGGER = getLogger(StreamingEntitySender.class);

  public static final int ENTITY_STREAMING_BUFFER_SIZE = 8192;
  private static final int END_OF_STREAM = -1;

  private final ChannelHandlerContext ctx;
  private final int bufferSize;
  private final Runnable beforeWrite;
  private final StatusCallback statusCallback;
  private final InputStream contentAsInputStream;
  private final OptionalLong entityLength;
  private final AtomicLong bytesAlreadySent;

  private final long initialNanos;
  private final AtomicBoolean isAlreadyScheduledToIO;
  private final Executor ioExecutor;

  public StreamingEntitySender(HttpEntity entity, ChannelHandlerContext ctx, Runnable beforeWrite,
                               StatusCallback statusCallback, Executor ioExecutor) {
    this.ctx = ctx;
    this.contentAsInputStream = entity.getContent();
    this.entityLength = entity.getBytesLength();
    this.bufferSize = calculateBufferSize();
    this.beforeWrite = beforeWrite;
    this.statusCallback = statusCallback;
    this.bytesAlreadySent = new AtomicLong(0);

    this.initialNanos = nanoTime();
    this.isAlreadyScheduledToIO = new AtomicBoolean(false);
    this.ioExecutor = ioExecutor;
  }

  public void sendNextChunk() throws IOException {
    if (bufferSize == 0) {
      sendEmptyContentAndFinish(contentAsInputStream);
      return;
    }

    if (shouldScheduleNextChunkToIO()) {
      boolean couldScheduleToIO = tryScheduleNextChunkHandlingToIO();
      if (couldScheduleToIO) {
        return;
      } else {
        LOGGER
            .debug("Tried to schedule next chunk handling to IO, but the task was rejected. It will be executed in the current thread.");
      }
    }

    byte[] buf = new byte[bufferSize];
    int bytesActuallyRead = readChunk(contentAsInputStream, buf);

    if (END_OF_STREAM == bytesActuallyRead) {
      // there is no more data because the end of the stream has been reached
      sendEmptyContentAndFinish(contentAsInputStream);
    } else {
      ByteBuf content = createBuffer(bytesActuallyRead);
      content.writeBytes(buf, 0, bytesActuallyRead);
      if (weKnowItIsTheLastChunk(bytesActuallyRead, bytesAlreadySent.get())) {
        sendBufferWithPromise(new DefaultLastHttpContent(content), finishStreamingPromise(contentAsInputStream));
      } else {
        sendBufferWithPromise(new DefaultHttpContent(content), sendNextChunkPromise());
      }
      bytesAlreadySent.addAndGet(bytesActuallyRead);
    }
  }

  private boolean shouldScheduleNextChunkToIO() {
    return NANOSECONDS.toMillis(nanoTime() - initialNanos) >= 50 && !isAlreadyScheduledToIO.get();
  }

  private boolean tryScheduleNextChunkHandlingToIO() {
    try {
      ioExecutor.execute(() -> {
        try {
          LOGGER.debug("Scheduling next chunk handling to IO");
          sendNextChunk();
        } catch (IOException e) {
          statusCallback.onFailure(e);
        }
      });
      isAlreadyScheduledToIO.set(true);
      return true;
    } catch (RejectedExecutionException ree) {
      return false;
    }
  }

  private boolean weKnowItIsTheLastChunk(int bytesToSend, long bytesAlreadySent) {
    if (!entityLength.isPresent()) {
      return false;
    }
    return bytesAlreadySent + bytesToSend >= entityLength.getAsLong();
  }

  private int calculateBufferSize() {
    return min(toIntExact(entityLength.orElse(ENTITY_STREAMING_BUFFER_SIZE)), ENTITY_STREAMING_BUFFER_SIZE);
  }

  private ByteBuf createBuffer(int size) {
    return ctx.alloc().buffer(size, size);
  }

  private ChannelPromise createPromise(ChannelFutureListener listener) {
    ChannelPromise promise = ctx.newPromise();
    promise.addListener(listener);
    return promise;
  }

  private void sendEmptyContentAndFinish(InputStream contentAsInputStream) {
    beforeWrite.run();
    ctx.writeAndFlush(EMPTY_LAST_CONTENT, finishStreamingPromise(contentAsInputStream));
  }

  private ChannelPromise finishStreamingPromise(InputStream stream) {
    return createPromise(new FinishStreamingListener(stream, statusCallback));
  }

  private ChannelPromise sendNextChunkPromise() {
    ChannelPromise promise = ctx.newPromise();
    promise.addListener(new SendNextChunkListener(this, statusCallback));
    return promise;
  }

  private void sendBufferWithPromise(HttpContent content, ChannelPromise promise) {
    beforeWrite.run();
    ctx.writeAndFlush(content, promise);
  }

  private static int readChunk(InputStream contentAsInputStream, byte[] buf) throws IOException {
    try {
      return contentAsInputStream.read(buf);
    } catch (IllegalStateException exception) {
      if ("Buffer is closed".equals(exception.getMessage())) {
        // TODO: Does it always mean that there is no more data or we can be silencing some error?
        return END_OF_STREAM;
      } else {
        throw exception;
      }
    }
  }
}
