/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;

import static junit.framework.TestCase.fail;

import static org.junit.jupiter.params.provider.Arguments.of;

import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.server.async.ResponseStatusCallback;
import org.mule.service.http.netty.impl.service.NettyHttpServiceImplementation;
import org.mule.tck.SimpleUnitTestSupportSchedulerService;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.util.stream.Stream;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.api.extension.ParameterContext;
import org.junit.jupiter.api.extension.ParameterResolutionException;
import org.junit.jupiter.api.extension.ParameterResolver;
import org.junit.jupiter.params.ParameterizedClass;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

/**
 * Base class for all HTTP service functional tests in JUnit 5. The service implementation will be loaded based on it's class
 * name, allowing tests to customize it.
 */
@ParameterizedClass
@MethodSource("serviceNames")
public abstract class AbstractHttpServiceTestCase extends AbstractMuleTestCase implements ParameterResolver {

  public String serviceToLoad;

  protected HttpService service;
  private SchedulerService schedulerService;

  public AbstractHttpServiceTestCase(String serviceToLoad) {
    this.serviceToLoad = serviceToLoad;
  }

  public static Stream<Arguments> serviceNames() {
    return Stream.of(of(NettyHttpServiceImplementation.class.getName()));
  }

  @BeforeEach
  public void createServices() throws Exception {
    schedulerService = getSchedulerService();
    service = (HttpService) Class.forName(serviceToLoad, true, this.getClass().getClassLoader())
        .getConstructor(SchedulerService.class)
        .newInstance(schedulerService);
    startIfNeeded(service);
  }

  protected SchedulerService getSchedulerService() {
    return new SimpleUnitTestSupportSchedulerService();
  }

  @AfterEach
  public void closeServices() throws Exception {
    if (service != null) {
      stopIfNeeded(service);
    }
    stopIfNeeded(schedulerService);
  }

  @Override
  public boolean supportsParameter(ParameterContext parameterContext, ExtensionContext extensionContext)
      throws ParameterResolutionException {
    return parameterContext.getParameter().getType().equals(String.class) &&
        parameterContext.getParameter().getName().equals("serviceToLoad");
  }

  @Override
  public Object resolveParameter(ParameterContext parameterContext, ExtensionContext extensionContext)
      throws ParameterResolutionException {
    return serviceToLoad;
  }

  public static class IgnoreResponseStatusCallback implements ResponseStatusCallback {

    @Override
    public void responseSendFailure(Throwable throwable) {
      fail(throwable.getMessage());
    }

    @Override
    public void responseSendSuccessfully() {
      // Do nothing
    }
  }
}
