/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.mule.service.http.test.netty.AllureConstants.HTTP_2;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.tck.junit5.DynamicPort;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Feature;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

@Feature(HTTP_2)
class Http2ClientServerRequestsTestCase extends AbstractHttpServiceTestCase {

  @DynamicPort(systemProperty = "serverPort")
  Integer serverPort;

  private HttpServer httpServer;
  private HttpClient httpClient;

  private final CompletableFuture<HttpRequestContext> seenRequestCtx = new CompletableFuture<>();

  public Http2ClientServerRequestsTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  void setUp() throws Exception {
    httpServer = createServer();
    httpClient = createClient();
  }

  @AfterEach
  void tearDown() {
    httpClient.stop();
    httpServer.stop().dispose();
  }

  @Test
  void sendGet() throws ExecutionException, InterruptedException {
    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    httpClient.sendAsync(request).get();
    var requestInServer = seenRequestCtx.get().getRequest();
    assertThat(requestInServer.getMethod(), is("GET"));
    assertThat(requestInServer.getPath(), is("/test"));
    assertThat(requestInServer.getProtocol().asString(), is("HTTP/2"));
  }

  @Test
  void sendGetWithQueryParams() throws ExecutionException, InterruptedException {
    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .addQueryParam("queryParam1", "value1.1")
        .addQueryParam("queryParam1", "value1.2")
        .addQueryParam("queryParam2", "value2")
        .build();

    httpClient.sendAsync(request).get();
    var requestInServer = seenRequestCtx.get().getRequest();
    assertThat(requestInServer.getQueryParams().getAll("queryParam1"), contains("value1.1", "value1.2"));
    assertThat(requestInServer.getQueryParams().get("queryParam2"), is("value2"));
  }

  @Test
  void sendGetWithHeaders() throws ExecutionException, InterruptedException {
    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .addHeader("header1", "value1")
        .addHeader("header2", "value2")
        .build();

    httpClient.sendAsync(request).get();
    var requestInServer = seenRequestCtx.get().getRequest();
    assertThat(requestInServer.getHeaders().get("header1"), is("value1"));
    assertThat(requestInServer.getHeaders().get("header2"), is("value2"));
  }

  @Test
  void sendGetWithMultimapHeaders() throws ExecutionException, InterruptedException {
    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .addHeader("header1", "value1.1")
        .addHeader("header1", "value1.2")
        .addHeader("header2", "value2")
        .build();

    httpClient.sendAsync(request).get();
    var requestInServer = seenRequestCtx.get().getRequest();
    assertThat(requestInServer.getHeaders().getAll("header1"), contains("value1.1", "value1.2"));
    assertThat(requestInServer.getHeaders().get("header2"), is("value2"));
  }

  private HttpClient createClient() throws CreateException {
    var client = service.getClientFactory().create(new HttpClientConfiguration.Builder()
        .setName("HTTP/2 Client")
        .setHttp2Config(new Http2ProtocolConfig(true))
        .setTlsContextFactory(TlsContextFactory.builder().trustStorePath("trustStore")
            .trustStorePassword("mulepassword").insecureTrustStore(true).build())
        .build());
    client.start();
    return client;
  }

  private HttpServer createServer()
      throws ServerCreationException, IOException, CreateException {
    var server = service.getServerFactory().create(new HttpServerConfiguration.Builder()
        .setName("HTTP/2 Server")
        .setHost("localhost")
        .setPort(serverPort)
        .setHttp2Config(new Http2ProtocolConfig(true))
        .setTlsContextFactory(TlsContextFactory.builder()
            .keyStorePath("serverKeystore")
            .keyStorePassword("mulepassword").keyAlias("muleserver").keyPassword("mulepassword").keyStoreAlgorithm("PKIX")
            .build())
        .build());

    server.start();
    server.addRequestHandler("/test", (ctx, callback) -> {
      seenRequestCtx.complete(ctx);
      callback.responseReady(HttpResponse.builder().build(), new NoOpResponseStatusCallback());
    });

    return server;
  }
}
