/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.functional.junit4.matchers.ThrowableCauseMatcher.hasCause;

import static java.lang.String.format;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.sameInstance;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.Http1ProtocolConfig;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerAlreadyExistsException;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.runtime.http.api.server.ServerNotFoundException;
import org.mule.service.http.netty.impl.server.HttpServerConnectionManager;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.util.function.Supplier;

import io.qameta.allure.Issue;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

public class HttpServerConnectionManagerTestCase extends AbstractMuleTestCase {

  public static final String TEST_HOST = "localhost";

  @Rule
  public DynamicPort dynamicPort = new DynamicPort("port");

  private HttpServerConnectionManager httpServerConnectionManager;
  private HttpServerConfiguration mockConfiguration;
  private Supplier<HttpServerConfiguration> mockConfigurationSupplier;
  private final Supplier<Long> shutdownTimeout = () -> 5000L;

  @Before
  public void setup() {
    httpServerConnectionManager = new HttpServerConnectionManager(null);

    TlsContextFactory tlsContextFactory = TlsContextFactory.builder().buildDefault();

    mockConfigurationSupplier = mock(Supplier.class);
    mockConfiguration = mock(HttpServerConfiguration.class);
    when(mockConfiguration.getName()).thenReturn("mockServerName");
    when(mockConfiguration.getHost()).thenReturn(TEST_HOST);
    when(mockConfiguration.getPort()).thenReturn(dynamicPort.getNumber());
    when(mockConfiguration.getTlsContextFactory()).thenReturn(tlsContextFactory);
    when(mockConfiguration.getHttp1Config()).thenReturn(new Http1ProtocolConfig(true));
    when(mockConfiguration.getHttp2Config()).thenReturn(new Http2ProtocolConfig(true));
    doReturn(mockConfiguration).when(mockConfigurationSupplier).get();
  }

  @Test
  public void getOrCreateServerReturnsSameInstanceIfCalledTwiceWithSameName() throws ServerCreationException {
    HttpServer firstCallServer =
        httpServerConnectionManager.getOrCreateServer("ServerName", "AppName", mockConfigurationSupplier, shutdownTimeout);
    HttpServer secondCallServer =
        httpServerConnectionManager.getOrCreateServer("ServerName", "AppName", mockConfigurationSupplier, shutdownTimeout);

    assertThat(secondCallServer, is(sameInstance(firstCallServer)));
  }

  @Test
  public void getOrCreateServerCallsConfigurationSupplierOnlyOnceIfCalledTwiceWithSameName() throws ServerCreationException {
    httpServerConnectionManager.getOrCreateServer("ServerName", "AppName", mockConfigurationSupplier, shutdownTimeout);
    httpServerConnectionManager.getOrCreateServer("ServerName", "AppName", mockConfigurationSupplier, shutdownTimeout);

    verify(mockConfigurationSupplier, times(1)).get();
  }

  @Test
  public void getOrCreateServerReturnsDifferentInstancesIfCalledTwiceWithDifferentNames() throws ServerCreationException {
    HttpServer firstCallServer =
        httpServerConnectionManager.getOrCreateServer("SomeServerName", "AppName", mockConfigurationSupplier, shutdownTimeout);
    HttpServer secondCallServer =
        httpServerConnectionManager.getOrCreateServer("OtherServerName", "AppName", mockConfigurationSupplier, shutdownTimeout);

    assertThat(secondCallServer, is(not(sameInstance(firstCallServer))));
  }

  @Test
  public void getOrCreateServerCallsConfigurationSupplierTwiceIfCalledTwiceWithDifferentNames()
      throws ServerCreationException {
    httpServerConnectionManager.getOrCreateServer("SomeServerName", "AppName", mockConfigurationSupplier, shutdownTimeout);
    httpServerConnectionManager.getOrCreateServer("OtherServerName", "AppName", mockConfigurationSupplier, shutdownTimeout);

    verify(mockConfigurationSupplier, times(2)).get();
  }

  @Test
  public void serverNameCantBeNull() {
    var thrown =
        assertThrows(IllegalArgumentException.class,
                     () -> httpServerConnectionManager.getOrCreateServer(null, "AppName", mockConfigurationSupplier,
                                                                         shutdownTimeout));
    assertThat(thrown.getMessage(), containsString("Server name can't be null"));
  }

  @Test
  public void suppliedConfigurationCantBeNull() {
    when(mockConfigurationSupplier.get()).thenReturn(null);

    var thrown =
        assertThrows(ServerCreationException.class,
                     () -> httpServerConnectionManager.getOrCreateServer("SomeName", "AppName", mockConfigurationSupplier,
                                                                         shutdownTimeout));
    assertThat(thrown.getCause(), hasCause(instanceOf(IllegalArgumentException.class)));
    assertThat(thrown.getMessage(),
               containsString("Could not create server: java.lang.IllegalArgumentException: Server configuration can't be null"));
  }

  @Test
  @Issue("W-15784799")
  public void ifAServerWasCreatedThenItCanBeLookedUp() throws ServerCreationException, ServerNotFoundException {
    httpServerConnectionManager.create(mockConfiguration, "AppName", shutdownTimeout);
    HttpServer lookedUp = httpServerConnectionManager.lookup("mockServerName", "AppName");
    assertThat(lookedUp, is(notNullValue()));
  }

  @Test
  @Issue("W-15784799")
  public void isAServerIsAlreadyCreatedThenCreatingItAgainFails() throws ServerCreationException {
    httpServerConnectionManager.create(mockConfiguration, "AppName", shutdownTimeout);

    String expectedErrorMessage =
        format("Could not create server: A server in port(%s) already exists for host(127.0.0.1) or one overlapping it (0.0.0.0).",
               dynamicPort.getNumber());

    var thrown = assertThrows(ServerAlreadyExistsException.class,
                              () -> httpServerConnectionManager.create(mockConfiguration, "AppName", shutdownTimeout));
    assertThat(thrown.getMessage(), containsString(expectedErrorMessage));
  }

  @Test
  @Issue("W-15795503")
  public void lifecycleCantBeAppliedIfTheServerIsLookedUp() throws ServerCreationException, ServerNotFoundException, IOException {
    HttpServer created = httpServerConnectionManager.create(mockConfiguration, "AppName", shutdownTimeout).start();
    HttpServer lookedUp = httpServerConnectionManager.lookup("mockServerName", "AppName");

    lookedUp.stop();
    assertThat(created.isStopped(), is(false));
    assertThat(lookedUp.isStopped(), is(false));

    created.stop();
    assertThat(created.isStopped(), is(true));
    assertThat(lookedUp.isStopped(), is(true));
  }
}
