/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.common.server.sse;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpHeaders.Names.ACCESS_CONTROL_ALLOW_ORIGIN;
import static org.mule.runtime.http.api.HttpHeaders.Names.CACHE_CONTROL;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.runtime.http.api.HttpHeaders.Names.PRAGMA;
import static org.mule.runtime.http.api.HttpHeaders.Names.TRANSFER_ENCODING;
import static org.mule.runtime.http.api.HttpHeaders.Values.CHUNKED;
import static org.mule.runtime.http.api.HttpHeaders.Values.NO_CACHE;
import static org.mule.runtime.http.api.HttpHeaders.Values.TEXT_EVENT_STREAM;

import static java.nio.charset.StandardCharsets.UTF_8;

import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.sse.server.SseClient;
import org.mule.runtime.http.api.sse.server.SseClientConfig;

import java.io.Writer;
import java.util.concurrent.CompletableFuture;

/**
 * Starts an SSE response (sends the header) and gives the caller a {@link SseClient} to send the corresponding events. The user
 * is in charge to {@link SseClient#close() close} the client reference.
 */
public class SseResponseStarter {

  private static final String SSE_CACHE_CONTROL = "no-cache, no-store, max-age=0, must-revalidate";
  private static final String CORS_WILDCARD = "*";

  /**
   * Sends the header part of the response and gives you a client reference.
   *
   * @param config           the config to create the {@link SseClient}.
   * @param responseCallback the {@link HttpResponseReadyCallback} used to send the response.
   * @return the {@link SseClient} to send the corresponding events. The user has to close it after use.
   */
  public SseClient startResponse(SseClientConfig config, HttpResponseReadyCallback responseCallback) {
    var responseBuilder = HttpResponse.builder()
        .statusCode(OK.getStatusCode()).reasonPhrase(OK.getReasonPhrase())
        .addHeader(CONTENT_TYPE, TEXT_EVENT_STREAM)
        .addHeader(CACHE_CONTROL, SSE_CACHE_CONTROL)
        .addHeader(PRAGMA, NO_CACHE)
        .addHeader(TRANSFER_ENCODING, CHUNKED)
        .addHeader(ACCESS_CONTROL_ALLOW_ORIGIN, CORS_WILDCARD)
        .entity(new EmptyHttpEntity());

    config.getResponseCustomizerConsumer().accept(new SseResponseCustomizerImpl(responseBuilder));

    CompletableFuture<Void> responseSentFuture = new CompletableFuture<>();
    Writer bodyWriter =
        responseCallback.startResponse(responseBuilder.build(), new FutureCompleterCallback(responseSentFuture), UTF_8);
    return new SseClientImpl(config, bodyWriter, responseSentFuture);
  }
}
