/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.utils;

import static io.netty.handler.codec.http.HttpResponseStatus.OK;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;
import org.mule.service.http.netty.impl.message.content.StreamedMultipartHttpEntity;
import org.mule.service.http.netty.impl.util.HttpResponseCreatorUtils;
import org.mule.service.http.test.common.AbstractHttpTestCase;

import java.io.ByteArrayInputStream;
import java.io.InputStream;

import io.netty.handler.codec.http.HttpHeaders;
import io.netty.handler.codec.http.HttpResponse;
import io.qameta.allure.Issue;
import org.junit.Before;
import org.junit.Test;
import reactor.netty.http.client.HttpClientResponse;

public class HttpResponseCreatorUtilsTestCase extends AbstractHttpTestCase {

  private HttpResponseCreatorUtils httpResponseCreatorUtils;

  @Before
  public void setUp() {
    httpResponseCreatorUtils = new HttpResponseCreatorUtils();
  }


  @Test
  @Issue("W-17843326")
  public void testCreateEntity_withMultipartContent() {
    InputStream inputStream = new ByteArrayInputStream("test content".getBytes());
    String contentType = "multipart/mixed";
    String contentLength = "100";
    int statusCode = 200;

    HttpEntity entity = httpResponseCreatorUtils.createEntity(inputStream, contentType, contentLength, statusCode);
    assertThat(entity, is(instanceOf(StreamedMultipartHttpEntity.class)));
  }

  @Test
  @Issue("W-17843326")
  public void testCreateEntity_withContentLength() {
    InputStream inputStream = new ByteArrayInputStream("test content".getBytes());

    HttpEntity entity = httpResponseCreatorUtils.createEntity(inputStream, "text/plain", "100", 200);
    assertThat(entity, is(instanceOf(InputStreamHttpEntity.class)));
  }

  @Test
  @Issue("W-17843326")
  public void testCreateEntity_withEmptyResponse() {
    InputStream inputStream = new ByteArrayInputStream("".getBytes());

    HttpEntity entity = httpResponseCreatorUtils.createEntity(inputStream, "text/plain", "0", 204);

    assertThat(entity, is(instanceOf(EmptyHttpEntity.class)));
  }

  @Test
  @Issue("W-17843326")
  public void testBuildResponseHeaders_forHttpResponse() {
    HttpResponse response = mock(HttpResponse.class);
    HttpHeaders headers = mock(HttpHeaders.class);
    when(response.status()).thenReturn(mock(io.netty.handler.codec.http.HttpResponseStatus.class));
    when(response.headers()).thenReturn(headers);
    when(response.status().code()).thenReturn(200);
    when(response.status().reasonPhrase()).thenReturn("OK");

    HttpResponseBuilder builder = httpResponseCreatorUtils.buildResponseHeaders(response);

    assertThat(builder.getStatusCode(), is(OK.code()));
    verify(headers, times(1)).names();
  }

  @Test
  @Issue("W-17843326")
  public void testBuildResponseHeaders_forHttpClientResponse() {
    HttpClientResponse response = mock(HttpClientResponse.class);
    HttpHeaders headers = mock(HttpHeaders.class);
    when(response.status()).thenReturn(mock(io.netty.handler.codec.http.HttpResponseStatus.class));
    when(response.responseHeaders()).thenReturn(headers);
    when(response.status().code()).thenReturn(200);
    when(response.status().reasonPhrase()).thenReturn("OK");

    HttpResponseBuilder builder = httpResponseCreatorUtils.buildResponseHeaders(response);

    assertThat(builder.getStatusCode(), is(OK.code()));
    verify(headers, times(1)).names();
  }

  @Test
  @Issue("W-17843326")
  public void testExtractHeader_forHttpResponse() {
    HttpResponse response = mock(HttpResponse.class);
    HttpHeaders headers = mock(HttpHeaders.class);
    when(response.headers()).thenReturn(headers);
    when(headers.contains("Content-Type")).thenReturn(true);
    when(headers.get("Content-Type")).thenReturn("application/json");

    String header = httpResponseCreatorUtils.extractHeader(response, "Content-Type");

    assertThat(header, is("application/json"));
  }

  @Test
  @Issue("W-17843326")
  public void testExtractHeader_forHttpClientResponse() {
    HttpClientResponse response = mock(HttpClientResponse.class);
    HttpHeaders headers = mock(HttpHeaders.class);
    when(response.responseHeaders()).thenReturn(headers);
    when(headers.contains("Content-Length")).thenReturn(true);
    when(headers.get("Content-Length")).thenReturn("100");

    String header = httpResponseCreatorUtils.extractHeader(response, "Content-Length");

    assertThat(header, is("100"));
  }
}
