/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static java.lang.Thread.currentThread;

import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.tck.junit5.DynamicPort;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;

/**
 * Sets up the service and a server that can be configured by subclasses to return specific responses, handling all lifecycle.
 */
public abstract class AbstractHttpClientTestCase extends AbstractHttpServiceTestCase {

  protected static final int TIMEOUT = 10000;

  @DynamicPort(systemProperty = "port")
  Integer port;

  protected HttpServer server;

  public AbstractHttpClientTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  public void setUp() throws Exception {
    server = service.getServerFactory().create(getServerConfigurationBuilder().build());
    server.start();
    RequestHandler requestHandler = getRequestHandler();

    var tccl = currentThread().getContextClassLoader();
    try {
      currentThread().setContextClassLoader(requestHandler.getContextClassLoader());
      server.addRequestHandler("/*", requestHandler);
    } finally {
      currentThread().setContextClassLoader(tccl);
    }
  }

  /**
   * @return the basic configuration of the test server so subclasses can customize it
   */
  protected HttpServerConfiguration.Builder getServerConfigurationBuilder() throws Exception {
    return new HttpServerConfiguration.Builder().setHost("localhost").setPort(port).setName("client-test-server");
  }

  /**
   * @param request the {@link HttpRequest} received by the server
   * @return the {@link HttpResponse} to return
   */
  protected abstract HttpResponse setUpHttpResponse(HttpRequest request);

  /**
   * @return the server's URI
   */
  protected String getUri() {
    return "http://localhost:" + port;
  }

  @AfterEach
  public void tearDown() throws Exception {
    if (server != null) {
      server.stop();
      server.dispose();
    }
  }

  protected HttpRequestOptions getDefaultOptions(int responseTimeout) {
    return HttpRequestOptions.builder().responseTimeout(responseTimeout).build();
  }

  protected RequestHandler getRequestHandler() {
    return (requestContext, responseCallback) -> responseCallback
        .responseReady(setUpHttpResponse(requestContext.getRequest()),
                       new IgnoreResponseStatusCallback());
  }

}
