/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.apache.hc.client5.http.async.methods.SimpleHttpRequest.create;
import static org.apache.hc.client5.http.impl.async.HttpAsyncClients.createHttp2Default;
import static org.apache.hc.core5.http.ContentType.TEXT_PLAIN;
import static org.apache.hc.core5.http.HttpVersion.HTTP_2;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.http.api.Http1ProtocolConfig;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.test.common.server.AbstractHttpServerTestCase;
import org.mule.service.http.test.netty.AllureConstants;

import java.io.IOException;

import io.qameta.allure.Feature;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.junitpioneer.jupiter.Issue;

@Feature(AllureConstants.HTTP_2)
class Http2SimpleServerTestCase extends AbstractHttpServerTestCase {

  public static final String H2_PRIOR_ECHO_ENDPOINT = "/h2-prior-echo";

  public Http2SimpleServerTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected String getServerName() {
    return "HTTP/2 Server";
  }

  @Override
  protected HttpServerConfiguration.Builder configureServer(HttpServerConfiguration.Builder builder) {
    return builder
        .setHttp1Config(new Http1ProtocolConfig(false))
        .setHttp2Config(new Http2ProtocolConfig(true));
  }

  @BeforeEach
  void setUp() throws Exception {
    setUpServer();
    server.addRequestHandler(H2_PRIOR_ECHO_ENDPOINT, (reqCtx, callback) -> {
      var req = reqCtx.getRequest();
      String body = extractBodyAsString(req);
      String queryParams = req.getQueryParams().toString();
      String uriString = req.getUri().toString();

      var echoBody = new StringHttpEntity("""
          Request{
            body: %s,
            uri: %s,
            queryParams: %s
          }
          """.formatted(body, uriString, queryParams));
      var res = HttpResponse.builder()
          .statusCode(200)
          .entity(echoBody)
          .build();
      callback.responseReady(res, new IgnoreResponseStatusCallback());
    });
  }

  @Test
  @Disabled("TODO (W-19814153): Schedule the request handler to IO")
  void echoPostToSimpleHttp2PriorKnowledgeServer() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      var request = create("POST", urlForPath(H2_PRIOR_ECHO_ENDPOINT));
      request.setBody("Hello world!", TEXT_PLAIN);
      var response = h2Client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(200));
      assertThat(response.getVersion(), is(HTTP_2));
    }
  }

  @Test
  void getToSimpleHttp2PriorKnowledgeServer() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      var request = create("GET", urlForPath(H2_PRIOR_ECHO_ENDPOINT));
      var response = h2Client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(200));
      assertThat(response.getVersion(), is(HTTP_2));
    }
  }

  @Test
  @Issue("W-19802358")
  void getToSimpleHttp2PriorKnowledgeServerWithQueryParams() throws Exception {
    try (var h2Client = createHttp2Default()) {
      h2Client.start();

      var request = create("GET", urlForPath(H2_PRIOR_ECHO_ENDPOINT + "?name1=value1&name2=value2"));
      var response = h2Client.execute(request, new IgnoreFutureCallback<>()).get();
      assertThat(response.getCode(), is(200));
      assertThat(response.getVersion(), is(HTTP_2));
      assertThat(response.getBodyText(), allOf(
                                               // The HTTP API offers the query params in its own method getQueryParams
                                               containsString("queryParams: MultiMap{[name1=[value1], name2=[value2]]}"),

                                               // But the HTTP Connector needs the query params present in the uri string to build
                                               // the HttpRequestAttributes
                                               containsString("uri: " + H2_PRIOR_ECHO_ENDPOINT + "?name1=value1&name2=value2")));
    }
  }

  // This method consumes the full body SYNCHRONOUSLY.
  private String extractBodyAsString(HttpRequest request) {
    try {
      return new String(request.getEntity().getBytes());
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
}
