/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import java.util.List;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.MessageToMessageDecoder;
import io.netty.handler.codec.http.DefaultFullHttpRequest;
import io.netty.handler.codec.http.EmptyHttpHeaders;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpObject;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.LastHttpContent;
import io.netty.util.ReferenceCounted;

/**
 * Decoder that aggregates requests when they consist only on a {@link HttpRequest} opening the message (not a
 * {@link FullHttpRequest}), and a {@link LastHttpContent} with all the content. Aggregating those objects into a single one is
 * useful to optimize the logic of {@link ForwardingToListenerHandler} that will handle the request synchronously when it receives
 * a {@link FullHttpRequest}.
 */
public class ConditionalRequestAggregator extends MessageToMessageDecoder<HttpObject> {

  private HttpRequest requestHeader;

  @Override
  protected void decode(ChannelHandlerContext channelHandlerContext, HttpObject httpObject, List<Object> list) throws Exception {
    if (httpObject instanceof FullHttpRequest) {
      // If it's already a full message, just forward it to the pipeline.
      stepAside(httpObject, list);
      return;
    }

    if (httpObject.decoderResult().isFailure()) {
      // If failure, somebody else will take care.
      stepAside(httpObject, list);
      return;
    }

    if (httpObject instanceof HttpRequest httpRequest) {
      // It starts the request, and it isn't a full request, so save it for later and return.
      this.requestHeader = httpRequest;
      return;
    }

    if (requestHeader != null && httpObject instanceof LastHttpContent lastHttpContent) {
      // The request is just the header and the last content, we can aggregate it...
      lastHttpContent.retain();
      var aggregatedRequest = combine(requestHeader, lastHttpContent.content());
      requestHeader = null;
      list.add(aggregatedRequest);
      return;
    }

    // Not a simple case, do nothing...
    stepAside(httpObject, list);
  }

  // Builds a full request from the header part and the content
  private static FullHttpRequest combine(HttpRequest request, ByteBuf content) {
    return new DefaultFullHttpRequest(request.protocolVersion(),
                                      request.method(),
                                      request.uri(),
                                      content,
                                      request.headers(),
                                      EmptyHttpHeaders.INSTANCE);
  }

  // This method is used to leave the aggregator without effect. The next handler in the pipeline will see the same objects as if
  // the aggregator is not present.
  private void stepAside(HttpObject httpObject, List<Object> list) {
    if (requestHeader != null) {
      list.add(requestHeader);
      requestHeader = null;
    }

    if (httpObject instanceof ReferenceCounted referenceCounted) {
      referenceCounted.retain();
    }
    list.add(httpObject);
  }
}
