/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.mule.service.http.test.common.util.HttpMessageHeaderMatcher.hasHeader;
import static org.mule.service.http.test.netty.AllureConstants.HTTP_2;
import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_1_AND_2_COMPATIBILITY;
import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_2_CLEARTEXT;
import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_2_CLEARTEXT_PRIOR_KNOWLEDGE;
import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_2_WITH_SSL;
import static org.mule.service.http.test.netty.utils.TestUtils.sendRawRequestToServer;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.junit.internal.matchers.ThrowableCauseMatcher.hasCause;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;
import static org.junit.jupiter.api.Assertions.assertThrows;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.http.api.Http1ProtocolConfig;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.service.http.test.common.AbstractHttpServiceTestCase;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.tck.junit5.DynamicPort;

import java.io.IOException;
import java.util.concurrent.ExecutionException;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;

/**
 * Combinatory testing from HTTP/1 and HTTP/2 supporting client-server, with and without (simple) SSL.
 */
@Feature(HTTP_2)
class Http2ClientServerCombinationsTestCase extends AbstractHttpServiceTestCase {

  private static final String INVALID_HTTP2_SETTINGS_REQUEST = """
      GET /test HTTP/1.1\r
      Host: localhost\r
      Connection: Upgrade, HTTP2-Settings\r
      Upgrade: h2c\r
      HTTP2-Settings: AAAABAAAAAAA\r
      User-Agent: ObviouslyMe\r

      """;

  @DynamicPort(systemProperty = "serverPort")
  Integer serverPort;

  private HttpServer httpServer;
  private HttpClient httpClient;

  public Http2ClientServerCombinationsTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @AfterEach
  void tearDown() {
    if (httpClient != null) {
      httpClient.stop();
    }
    if (httpServer != null) {
      httpServer.stop().dispose();
    }
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  void http1OnlyCleartext()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, false, false);
    httpClient = createClient(true, false, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/1.1"));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  void http1OnlyWithSSL()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, false, true);
    httpClient = createClient(true, false, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/1.1"));
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  @Story(HTTP_2_CLEARTEXT_PRIOR_KNOWLEDGE)
  void http2OnlyCleartext()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(false, true, false);
    httpClient = createClient(false, true, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/2"));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  void http2OnlyWithSSL()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(false, true, true);
    httpClient = createClient(false, true, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/2"));
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  void bothProtocolsCleartext()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, true, false);
    httpClient = createClient(true, true, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/2"));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  void bothProtocolsWithSSL()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, true, true);
    httpClient = createClient(true, true, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/2"));
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientBothProtocolsServerHttp1OnlyCleartext()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, false, false);
    httpClient = createClient(true, true, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/1.1"));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientBothProtocolsServerHttp1OnlyWithSSL()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, false, true);
    httpClient = createClient(true, true, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/1.1"));
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  @Story(HTTP_2_CLEARTEXT_PRIOR_KNOWLEDGE)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  @Description("The client will try to establish a HTTP/1 connection and then upgrade it to HTTP/2, while the server expects prior-knowledge")
  void clientBothProtocolsServerHttp2OnlyCleartext() throws ServerCreationException, IOException, CreateException {
    httpServer = createServer(false, true, false);
    httpClient = createClient(true, true, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var future = httpClient.sendAsync(request);
    var thrown = assertThrows(ExecutionException.class, future::get);
    assertThat(thrown, hasCause(hasMessage(containsString("Remotely closed"))));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  @Story(HTTP_2_CLEARTEXT_PRIOR_KNOWLEDGE)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientBothProtocolsServerHttp2OnlyWithSSL()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(false, true, true);
    httpClient = createClient(true, true, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/2"));
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientHttp1OnlyServerBothProtocolsCleartext()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, true, false);
    httpClient = createClient(true, false, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/1.1"));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientHttp1OnlyServerBothProtocolsWithSSL()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, true, true);
    httpClient = createClient(true, false, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/1.1"));
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  @Story(HTTP_2_CLEARTEXT_PRIOR_KNOWLEDGE)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientHttp2OnlyServerBothProtocolsCleartext()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, true, false);
    httpClient = createClient(false, true, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/2"));
  }

  @Test
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  @Disabled("W-19895731")
  void upgradeHttp1ToHttp2Returns101() throws Exception {
    httpServer = createServer(true, true, false);
    String rawRequest = """
        GET /test HTTP/1.1\r
        Host: localhost\r
        Connection: Upgrade, HTTP2-Settings\r
        Upgrade: h2c\r
        HTTP2-Settings: AAEAABAAAAQAAAEAAP__AAU=\r
        User-Agent: ObviouslyMe\r

        """;

    String response = sendRawRequestToServer(rawRequest, "localhost", serverPort);
    assertThat(response, containsString("HTTP/2 101"));
  }

  @Test
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void upgradeWithIncorrectSettingsReturns400() throws Exception {
    httpServer = createServer(true, true, false);
    String response = sendRawRequestToServer(INVALID_HTTP2_SETTINGS_REQUEST, "localhost", serverPort);
    assertThat(response, containsString("HTTP/1.1 400"));
    assertThat(response, containsString("Upgrade request to HTTP2 failed"));
  }

  @Test
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void upgradeWithIncorrectSettingsButWithNoHttp2AvailableReturnsHttp1_200() throws Exception {
    httpServer = createServer(true, false, false);
    String response = sendRawRequestToServer(INVALID_HTTP2_SETTINGS_REQUEST, "localhost", serverPort);
    assertThat(response, containsString("HTTP/1.1 200"));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientHttp2OnlyServerBothProtocolsWithSSL()
      throws ExecutionException, InterruptedException, ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, true, true);
    httpClient = createClient(false, true, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var response = httpClient.sendAsync(request).get();
    assertThat(response, hasHeader("X-Request-Protocol", "HTTP/2"));
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  @Story(HTTP_2_CLEARTEXT_PRIOR_KNOWLEDGE)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientHttp1OnlyServerHttp2OnlyCleartext() throws ServerCreationException, IOException, CreateException {
    httpServer = createServer(false, true, false);
    httpClient = createClient(true, false, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var future = httpClient.sendAsync(request);
    var thrown = assertThrows(ExecutionException.class, future::get);
    assertThat(thrown, hasCause(hasMessage(containsString("Remotely closed"))));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientHttp1OnlyServerHttp2OnlyWithSSL() throws ServerCreationException, IOException, CreateException {
    httpServer = createServer(false, true, true);
    httpClient = createClient(true, false, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var future = httpClient.sendAsync(request);
    var thrown = assertThrows(ExecutionException.class, future::get);
    assertThat(thrown, hasCause(hasMessage(containsString("Remotely closed"))));
  }

  @Test
  @Story(HTTP_2_CLEARTEXT)
  @Story(HTTP_2_CLEARTEXT_PRIOR_KNOWLEDGE)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientHttp2OnlyServerHttp1OnlyCleartext() throws ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, false, false);
    httpClient = createClient(false, true, false);

    var request = HttpRequest.builder()
        .uri("http://localhost:%d/test".formatted(serverPort))
        .build();

    var future = httpClient.sendAsync(request);
    var thrown = assertThrows(ExecutionException.class, future::get);
    assertThat(thrown, hasCause(hasMessage(containsString("Remotely closed"))));
  }

  @Test
  @Story(HTTP_2_WITH_SSL)
  @Story(HTTP_1_AND_2_COMPATIBILITY)
  void clientHttp2OnlyServerHttp1OnlyWithSSL() throws ServerCreationException, IOException, CreateException {
    httpServer = createServer(true, false, true);
    httpClient = createClient(false, true, true);

    var request = HttpRequest.builder()
        .uri("https://localhost:%d/test".formatted(serverPort))
        .build();

    var future = httpClient.sendAsync(request);
    var thrown = assertThrows(ExecutionException.class, future::get);
    assertThat(thrown, hasCause(hasMessage(containsString("Remotely closed"))));
  }

  private HttpClient createClient(boolean supportHttp1, boolean supportHttp2, boolean useTls) throws CreateException {
    HttpClientConfiguration.Builder configBuilder = new HttpClientConfiguration.Builder()
        .setName("HTTP/2 Client")
        .setHttp1Config(new Http1ProtocolConfig(supportHttp1))
        .setHttp2Config(new Http2ProtocolConfig(supportHttp2));

    if (useTls) {
      configBuilder.setTlsContextFactory(TlsContextFactory.builder().trustStorePath("trustStore")
          .trustStorePassword("mulepassword").insecureTrustStore(true).build());
    }

    var client = service.getClientFactory().create(configBuilder.build());

    client.start();
    return client;
  }

  private HttpServer createServer(boolean supportHttp1, boolean supportHttp2, boolean useTls)
      throws ServerCreationException, IOException, CreateException {
    HttpServerConfiguration.Builder configBuilder = new HttpServerConfiguration.Builder()
        .setName("HTTP/2 Server")
        .setHost("localhost")
        .setPort(serverPort)
        .setHttp1Config(new Http1ProtocolConfig(supportHttp1))
        .setHttp2Config(new Http2ProtocolConfig(supportHttp2));

    if (useTls) {
      configBuilder.setTlsContextFactory(TlsContextFactory.builder()
          .keyStorePath("serverKeystore")
          .keyStorePassword("mulepassword").keyAlias("muleserver").keyPassword("mulepassword").keyStoreAlgorithm("PKIX")
          .build());
    }

    var server = service.getServerFactory().create(configBuilder.build());

    server.start();
    server.addRequestHandler("/test", (ctx, callback) -> {
      callback.responseReady(HttpResponse.builder()
          .addHeader("X-Request-Protocol", ctx.getRequest().getProtocol().asString())
          .build(), new NoOpResponseStatusCallback());
    });

    return server;
  }
}
