/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;

import static org.mule.service.http.netty.impl.streaming.StreamingEntitySender.ENTITY_STREAMING_BUFFER_SIZE;

import static java.lang.Math.min;
import static java.lang.Math.toIntExact;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.entity.HttpEntity;

import java.io.IOException;
import java.io.InputStream;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import org.reactivestreams.Subscription;
import org.slf4j.Logger;
import reactor.core.CoreSubscriber;
import reactor.core.publisher.Flux;

/**
 * Converts the content of an HttpEntity into a Flux of ByteBuf by using an allocator buffer
 */
public class ChunkedHttpEntityPublisher extends Flux<ByteBuf> {

  private static final Logger LOGGER = getLogger(ChunkedHttpEntityPublisher.class);

  private final HttpEntity httpEntity;
  private final int bufferSize;

  public ChunkedHttpEntityPublisher(HttpEntity httpEntity) {
    this(httpEntity, ENTITY_STREAMING_BUFFER_SIZE);
  }

  public ChunkedHttpEntityPublisher(HttpEntity httpEntity, int bufferSize) {
    this.httpEntity = httpEntity;
    this.bufferSize = bufferSize;
  }

  @Override
  public void subscribe(CoreSubscriber<? super ByteBuf> actual) {
    actual.onSubscribe(new InputStreamSubscription(actual, httpEntity, bufferSize));
  }

  private static class InputStreamSubscription implements Subscription {

    private static final String HTTP_ENTITY_PRECONDITION =
        "If a HttpEntity returns true for isStreaming(), it has to provide a length in getBytesLength()";

    private final CoreSubscriber<? super ByteBuf> subscriber;
    private final HttpEntity httpEntity;
    private final int bufferSize;
    private final InputStream contentAsInputStream;

    InputStreamSubscription(CoreSubscriber<? super ByteBuf> subscriber, HttpEntity entity, int bufferSize) {
      this.subscriber = subscriber;
      this.httpEntity = entity;
      this.bufferSize = getBufferSize(entity, bufferSize);

      if (httpEntity.isStreaming()) {
        // Streaming entities can provide its content only once, so we call it in the constructor.
        this.contentAsInputStream = entity.getContent();
      } else {
        // We will call getBytes in this case.
        this.contentAsInputStream = null;
      }
    }

    private static int getBufferSize(HttpEntity entity, int bufferSize) {
      if (entity.isStreaming()) {
        // If the entity is shorter than the parameter bufferSize, we override it with the actual entity size.
        return min(toIntExact(entity.getBytesLength().orElse(bufferSize)), bufferSize);
      } else {
        // If not streaming, just return the entity length.
        return (int) entity.getBytesLength().orElseThrow(() -> new IllegalStateException(HTTP_ENTITY_PRECONDITION));
      }
    }

    @Override
    public void request(long requestedChunks) {
      try {
        if (httpEntity.isStreaming()) {
          sendRequestedChunksAndCompleteIfFullyConsumed(requestedChunks);
        } else {
          sendAllContentInOneNextAndComplete();
        }
      } catch (IOException e) {
        subscriber.onError(e);
      }
    }

    private void sendRequestedChunksAndCompleteIfFullyConsumed(long requestedChunks) throws IOException {
      boolean contentIsFullyConsumed = false;
      int sentChunks = 0;
      while (!contentIsFullyConsumed && sentChunks < requestedChunks) {
        byte[] buffer = new byte[bufferSize];
        int bytesActuallyRead = contentAsInputStream.read(buffer);
        contentIsFullyConsumed = (bytesActuallyRead <= 0);
        if (!contentIsFullyConsumed) {
          ByteBuf byteBuf = createByteBuf(buffer, bytesActuallyRead);
          subscriber.onNext(byteBuf);
          sentChunks += 1;
        }
      }

      if (contentIsFullyConsumed) {
        subscriber.onComplete();
      }
    }

    private void sendAllContentInOneNextAndComplete() throws IOException {
      byte[] bytesRead = httpEntity.getBytes();
      if (bytesRead.length > 0) {
        ByteBuf byteBuf = createByteBuf(bytesRead, bytesRead.length);
        subscriber.onNext(byteBuf);
      }
      subscriber.onComplete();
    }

    private static ByteBuf createByteBuf(byte[] bytesRead, int bufferSize) {
      return ByteBufAllocator.DEFAULT.buffer(bufferSize, bufferSize).writeBytes(bytesRead, 0, bufferSize);
    }

    @Override
    public void cancel() {
      try {
        if (contentAsInputStream != null) {
          contentAsInputStream.close();
        }
      } catch (IOException e) {
        LOGGER.warn("There was a problem while closing the request content stream", e);
      }
    }
  }
}
