/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client.auth;

import static org.mule.service.http.netty.impl.client.auth.AuthUtils.computeRealmURI;

import java.net.URI;
import java.util.concurrent.ThreadLocalRandom;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.codec.digest.DigestUtils;

public class Realm {

  private static final String DEFAULT_NC = "00000001";
  private String opaque;
  private String algorithm;
  private String username;
  private String password;
  private String realmName;
  private String nonce;
  private String cnonce;
  private URI uri;
  private String nc = DEFAULT_NC;
  private String qop;
  private String response;

  public Realm(String username, String password, String realmName, String nonce, String qop, String nc, String cnonce, URI uri,
               String opaque, String algorithm) {
    this.username = username;
    this.password = password;
    this.realmName = realmName;
    this.nonce = nonce;
    this.qop = qop;
    this.nc = nc;
    this.cnonce = cnonce;
    this.uri = uri;
    this.opaque = opaque;
    this.algorithm = algorithm;
  }

  public Realm(String username, String password, URI uri) {
    this.username = username;
    this.password = password;
    this.uri = uri;
  }

  public CharSequence getNonce() {
    return this.nonce;
  }

  public String getResponse() {
    return this.response;
  }

  public String getRealmName() {
    return this.realmName;
  }

  private String ha1() {
    return DigestUtils.md5Hex(this.username + ":" + this.realmName + ":" + this.password);
  }

  private String ha2(String digestUri, String method) {
    // // if qop is "auth" or is unspecified => A2 = Method ":" digest-uri-value
    // // if qop is "auth-int" => A2 = Method ":" digest-uri-value ":" H(entity-body)
    // sb.append(methodName).append(':').append(digestUri);
    // if ("auth-int".equals(qop)) {
    // // when qop == "auth-int", A2 = Method ":" digest-uri-value ":" H(entity-body)
    // // but we don't have the request body here
    // // we would need a new API
    // sb.append(':').append(EMPTY_ENTITY_MD5);
    //
    // } else if (qop != null && !"auth".equals(qop)) {
    // throw new UnsupportedOperationException("Digest qop not supported: " + qop);
    // }
    return DigestUtils.md5Hex(method + ":" + digestUri);
  }

  private String newResponse(URI uri, String method) {
    // when using preemptive auth, the request uri is missing
    if (uri != null) {
      String digestUri = AuthUtils.computeRealmURI(uri, false, false);

      String ha1 = ha1();
      String ha2 = ha2(digestUri, method);
      StringBuilder middlePart = new StringBuilder(nonce).append(':');
      if ("auth".equals(qop) || "auth-int".equals(qop)) {
        middlePart.append(nc).append(':').append(cnonce).append(':').append(qop);
      }
      this.response = DigestUtils.md5Hex(ha1 + ":" + middlePart + ":" + ha2);
    }
    return response;
  }

  private void newCnonce() {
    byte[] b = new byte[8];
    ThreadLocalRandom.current().nextBytes(b);
    cnonce = DigestUtils.md5Hex(b);
  }

  public static Realm newRealm(Realm realm, URI uri, String methodName) {
    Realm newRealm = new Realm(realm.username,
                               realm.password,
                               realm.realmName,
                               realm.nonce,
                               realm.qop,
                               realm.nc,
                               realm.cnonce,
                               uri,
                               realm.opaque,
                               realm.algorithm);

    // Avoid generating if not needed
    String response;
    if (!StringUtils.isBlank(realm.nonce)) {
      newRealm.newCnonce();
      response = newRealm.newResponse(uri, methodName);
    } else {
      response = realm.response;
    }
    newRealm.setResponse(response);
    return newRealm;
  }

  public static Realm newRealm(Realm realm, String realmName, String nonce, String qop, String opaque, String algorithm, URI uri,
                               String methodName) {
    Realm newRealm = new Realm(realm.username,
                               realm.password,
                               realmName,
                               nonce,
                               qop,
                               realm.nc,
                               realm.cnonce,
                               uri,
                               opaque,
                               algorithm);

    // Avoid generating if not needed
    String response;
    if (!StringUtils.isBlank(nonce)) {
      response = newRealm.newResponse(uri, methodName);
    } else {
      response = realm.response;
    }
    newRealm.setResponse(response);
    return newRealm;
  }

  private void setResponse(String response) {
    this.response = response;
  }

  public String getAlgorithm() {
    return this.algorithm;
  }

  public String getOpaque() {
    return this.opaque;
  }

  public String getQop() {
    return this.qop;
  }

  public String getNc() {
    return this.nc;
  }

  public String getCnonce() {
    return this.cnonce;
  }
}
