/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static io.netty.buffer.ByteBufUtil.writeUtf8;
import static io.netty.channel.ChannelFutureListener.CLOSE;
import static io.netty.handler.codec.http.HttpResponseStatus.BAD_REQUEST;
import static io.netty.util.ReferenceCountUtil.release;
import static org.slf4j.LoggerFactory.getLogger;

import java.util.concurrent.atomic.AtomicBoolean;

import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.handler.codec.http.DefaultFullHttpResponse;
import io.netty.handler.codec.http.HttpRequest;
import io.netty.handler.codec.http.HttpServerUpgradeHandler;
import org.slf4j.Logger;

/**
 * When an HTTP Upgrade request is invalid, Netty just leaves it in the pipeline to be processed by the next handlers. This
 * handler is intended to reject such failed upgrades.
 * <p>
 * If the implementer of an {@link HttpServerUpgradeHandler.UpgradeCodec} wants it to be rejected with a 400 status code, it has
 * to fire an event of type {@link FailedHttpUpgrade} with an error message that will be included as the body of the 400 response.
 * <p>
 * As Netty already decoded a non-successful upgrade request and leaved it in the pipeline, this handler will also filter any
 * further read requests after sending the 400 response.
 */
public class RejectFailedUpgradeHandler extends ChannelInboundHandlerAdapter {

  private static final Logger LOGGER = getLogger(RejectFailedUpgradeHandler.class);

  private final AtomicBoolean alreadyRejected = new AtomicBoolean(false);

  @Override
  public void userEventTriggered(ChannelHandlerContext ctx, Object evt) {
    if (evt instanceof FailedHttpUpgrade failedUpgrade) {
      alreadyRejected.set(true);
      sendBadRequest(ctx, failedUpgrade.upgradeRequest(), failedUpgrade.errorMessage());
    }
    ctx.fireUserEventTriggered(evt);
  }

  @Override
  public void channelRead(ChannelHandlerContext ctx, Object msg) throws Exception {
    if (alreadyRejected.get()) {
      LOGGER
          .debug("Skipping the processing of a request because it was a rejected upgrade. Processing it after this point would cause two responses to be sent for the same request.");
      release(msg);
      return;
    }
    ctx.fireChannelRead(msg);
  }

  private void sendBadRequest(ChannelHandlerContext ctx, HttpRequest request, String message) {
    var messageAsByteBuf = writeUtf8(ctx.alloc(), message);
    var badRequest = new DefaultFullHttpResponse(request.protocolVersion(), BAD_REQUEST, messageAsByteBuf);
    ctx.writeAndFlush(badRequest).addListener(CLOSE);
  }

  public record FailedHttpUpgrade(HttpRequest upgradeRequest, String errorMessage) {
  }
}
