/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.service;

import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.runtime.http.api.server.HttpServerFactory;
import org.mule.runtime.http.api.server.ServerCreationException;
import org.mule.runtime.http.api.server.ServerNotFoundException;
import org.mule.service.http.netty.impl.server.ContextHttpServerConnectionFactory;

import java.util.function.Supplier;

/**
 * Implementation of {@link HttpServerFactory} that receives the contextual information in the constructor to end up delegating
 * the {@link HttpServer} creation to a {@link ContextHttpServerConnectionFactory}.
 */
public class ContextHttpServerFactory implements HttpServerFactory {

  private final ContextHttpServerConnectionFactory delegate;
  private final Supplier<Long> shutdownTimeout;
  private final String context;
  private final String parentContext;

  public ContextHttpServerFactory(ContextHttpServerConnectionFactory delegate,
                                  String context, String parentContext,
                                  Supplier<Long> shutdownTimeout) {
    this.delegate = delegate;
    this.context = context;
    this.parentContext = parentContext;
    this.shutdownTimeout = shutdownTimeout;
  }

  @Override
  public HttpServer create(HttpServerConfiguration configuration) throws ServerCreationException {
    return delegate.create(configuration, context, shutdownTimeout);
  }

  @Override
  public HttpServer lookup(String name) throws ServerNotFoundException {
    try {
      return delegate.lookup(name, context);
    } catch (ServerNotFoundException e) {
      if (parentContext != null) {
        return delegate.lookup(name, parentContext);
      } else {
        throw e;
      }
    }
  }

  public String getContextName() {
    return context;
  }

  public String getParentContextName() {
    return parentContext;
  }
}
