/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.benchmark;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.concurrent.Executors.newFixedThreadPool;

import static io.netty.buffer.Unpooled.wrappedBuffer;
import static io.netty.util.ReferenceCountUtil.release;

import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;

import java.io.IOException;
import java.net.InetSocketAddress;

import io.netty.buffer.ByteBuf;
import io.netty.channel.embedded.EmbeddedChannel;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;

@State(Scope.Benchmark)
public class ServerPipelineBenchmarkState {

  private HttpServer server;

  public int port = 80;
  public EmbeddedChannel channel;

  private ByteBuf requestAsByteBuf;

  @Setup(Level.Trial)
  public void setUpTrial() throws IOException {
    var pool = newFixedThreadPool(10);
    var listenerRegistry = new HttpListenerRegistry();
    var serverConfig = new HttpServerConfiguration.Builder()
        .setName("BenchmarkServer")
        .setHost("localhost")
        .setPort(port)
        .build();
    var handler = new AcceptedConnectionChannelInitializer(listenerRegistry, serverConfig, null, pool);

    InetSocketAddress serverAddress = new InetSocketAddress(port);
    InetSocketAddress clientAddress = new InetSocketAddress(0);

    server = NettyHttpServer.builder()
        .withName("BenchmarkServer")
        .withServerAddress(serverAddress)
        .withShutdownTimeout(() -> 5000L)
        .withHttpListenerRegistry(listenerRegistry)
        .withClientChannelHandler(handler)
        .build();
    server.start();
    server.addRequestHandler("/", (request, response) -> {
      var res = HttpResponse.builder().statusCode(200).reasonPhrase("OK").build();
      response.responseReady(res, new NoOpResponseStatusCallback());
    }).start();

    channel = new EmbeddedChannelWithAddress(serverAddress, clientAddress, handler);

    requestAsByteBuf = wrappedBuffer("""
        GET / HTTP/1.1\r
        Content-Length: 0\r
        Host: localhost:80\r
        \r
        """.getBytes(UTF_8));
  }

  @TearDown(Level.Trial)
  public void tearDown() {
    release(requestAsByteBuf);
    channel.close();
    server.stop();
    server.dispose();
  }

  public ByteBuf getRawRequest() {
    // A retainedDuplicate will share the memory with the original buffer, but it will have its own indexes and refCnt.
    // The reference counter will be decreased by the embedded channel, because it will parse the data.
    // Therefore, nobody will consume the requestAsByteBuf, but each bench invocation will use this kind of "soft copy".
    return requestAsByteBuf.retainedDuplicate();
  }
}
