/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.client;


import static reactor.netty.ReactorNetty.format;

import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.handler.ssl.SslCloseCompletionEvent;
import reactor.util.Logger;
import reactor.util.Loggers;

/**
 * This handler takes care of doing the same as
 * {@link reactor.netty.channel.ChannelOperationsHandler#userEventTriggered(ChannelHandlerContext, Object)} (a.k.a.:
 * {@link reactor.netty.NettyPipeline#ReactiveBridge}).
 * <p>
 * The reason we need it for HTTP2 is that the {@link reactor.netty.NettyPipeline#ReactiveBridge} is removed from the main
 * pipeline by the {@link reactor.netty.http.client.HttpTrafficHandler#channelRead(ChannelHandlerContext, Object)}.
 * <p>
 * In that case, each stream will receive a child pipeline which will have a {@link reactor.netty.NettyPipeline#ReactiveBridge}
 * but if a connection receives a close_notify from the server when there are no ongoing streams (e.g.: after a connection idle
 * timeout), no one will handle the event.
 * <p>
 * This should probably be already handled by Reactor Netty, but it seems they did not take HTTP2 into account when they added it
 * <a href="https://github.com/reactor/reactor-netty/commit/2c9f79d60ef62e8ec99ba01feb1b2c0534fc431a">here</a>.
 */
class Http2SslCloseNotifyEventHandler extends ChannelInboundHandlerAdapter {

  static final Logger log = Loggers.getLogger(Http2SslCloseNotifyEventHandler.class);

  @Override
  public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
    if (evt instanceof SslCloseCompletionEvent sslCloseCompletionEvent) {
      // When a close_notify is received, the SSLHandler fires an SslCloseCompletionEvent.SUCCESS event,
      // so if the event is success and if the channel is still active (not closing for example),
      // then immediately close the channel.
      // see https://www.rfc-editor.org/rfc/rfc5246#section-7.2.1, which states that when receiving a close_notify,
      // then the connection must be closed down immediately.
      if (sslCloseCompletionEvent.isSuccess() && ctx.channel().isActive()) {
        if (log.isDebugEnabled()) {
          log.debug(format(ctx.channel(), "Received a TLS close_notify, closing the channel now."));
        }
        ctx.close();
        return;
      }
    }
    super.userEventTriggered(ctx, evt);
  }
}

