/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTP;
import static org.mule.runtime.http.api.HttpConstants.Protocol.HTTPS;
import static org.mule.runtime.http.api.server.MethodRequestMatcher.acceptAll;

import static java.lang.Math.min;
import static java.lang.Runtime.getRuntime;
import static java.lang.Thread.currentThread;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.MethodRequestMatcher;
import org.mule.runtime.http.api.server.PathAndMethodRequestMatcher;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.runtime.http.api.server.RequestHandlerManager;
import org.mule.runtime.http.api.server.ServerAddress;
import org.mule.runtime.http.api.server.ws.WebSocketHandler;
import org.mule.runtime.http.api.server.ws.WebSocketHandlerManager;
import org.mule.runtime.http.api.sse.server.SseClient;
import org.mule.runtime.http.api.sse.server.SseEndpointManager;
import org.mule.runtime.http.api.sse.server.SseRequestContext;
import org.mule.service.http.common.server.sse.SseHandlerManagerAdapter;
import org.mule.service.http.common.server.sse.SseRequestHandler;
import org.mule.service.http.netty.impl.server.util.DefaultServerAddress;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.netty.impl.util.HttpLoggingHandler;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.Collection;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.netty.bootstrap.ServerBootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.epoll.Epoll;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.epoll.EpollServerSocketChannel;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioServerSocketChannel;
import io.netty.handler.ssl.SslContext;
import org.slf4j.Logger;

public final class NettyHttpServer implements HttpServer {

  private static final Logger LOGGER = getLogger(NettyHttpServer.class);

  private static final int STOPPED = 0;
  private static final int STARTING = 1;
  private static final int STARTED = 2;
  private static final int STOPPING = 3;

  private int status = STOPPED;

  private final SslContext sslContext;
  private final ServerAddress serverAddress;
  private final Runnable onDisposeCallback;
  private final AcceptedConnectionChannelInitializer clientChannelHandler;
  private final HttpListenerRegistry httpListenerRegistry;
  private final WebSocketsHandlersRegistry webSocketsHandlersRegistry;
  private final int selectorsCount;
  private final Supplier<Long> shutdownTimeout;
  private Scheduler selectorsScheduler;
  private Channel serverChannel;
  private EventLoopGroup acceptorGroup;
  private EventLoopGroup workerGroup;

  private NettyHttpServer(ServerAddress serverAddress, HttpListenerRegistry httpListenerRegistry,
                          AcceptedConnectionChannelInitializer clientChannelHandler,
                          WebSocketsHandlersRegistry webSocketsHandlersRegistry, Scheduler selectorsScheduler, int selectorsCount,
                          Supplier<Long> shutdownTimeout, SslContext sslContext, Runnable onDisposeCallback) {
    this.serverAddress = serverAddress;
    this.httpListenerRegistry = httpListenerRegistry;
    this.clientChannelHandler = clientChannelHandler;
    this.webSocketsHandlersRegistry = webSocketsHandlersRegistry;
    this.selectorsScheduler = selectorsScheduler;
    this.selectorsCount = selectorsCount;
    this.shutdownTimeout = shutdownTimeout;
    this.sslContext = sslContext;
    this.onDisposeCallback = onDisposeCallback;

    this.acceptorGroup =
        Epoll.isAvailable() ? new EpollEventLoopGroup(1, selectorsScheduler) : new NioEventLoopGroup(1, selectorsScheduler);
    this.workerGroup =
        Epoll.isAvailable() ? new EpollEventLoopGroup(selectorsCount - 1, selectorsScheduler)
            : new NioEventLoopGroup(selectorsCount - 1, selectorsScheduler);
  }

  public static Builder builder() {
    return new Builder();
  }

  @Override
  public HttpServer start() throws IOException {
    status = STARTING;
    try {
      ServerBootstrap bootstrap = new ServerBootstrap();

      // TODO: With this method we can configure options such as SO_TIMEOUT, SO_KEEPALIVE, etc.
      bootstrap.option(ChannelOption.SO_BACKLOG, 1024);

      bootstrap.childOption(ChannelOption.ALLOW_HALF_CLOSURE, true);

      if (Epoll.isAvailable()) {
        bootstrap.group(acceptorGroup, workerGroup)
            .channel(EpollServerSocketChannel.class)
            .handler(HttpLoggingHandler.textual())
            .childHandler(clientChannelHandler);
      } else {
        bootstrap.group(acceptorGroup, workerGroup)
            .channel(NioServerSocketChannel.class)
            .handler(HttpLoggingHandler.textual())
            .childHandler(clientChannelHandler);
      }

      serverChannel = bootstrap.bind(serverAddress.getAddress(), serverAddress.getPort()).sync().channel();

      status = STARTED;
      LOGGER.info("HTTP Server is listening on address: {}", serverAddress);
      return this;
    } catch (InterruptedException e) {
      status = STOPPED;
      stop();
      currentThread().interrupt();
      throw new IOException(e);
    }
  }

  @Override
  public HttpServer stop() {
    status = STOPPING;
    if (serverChannel != null) {
      ChannelFuture channelFuture = serverChannel.close();
      clientChannelHandler.waitForConnectionsToBeClosed(shutdownTimeout.get(), MILLISECONDS);
      channelFuture.syncUninterruptibly();
      serverChannel = null; // Clear reference
    }
    status = STOPPED;
    return this;
  }


  @Override
  public void dispose() {
    if (!isStopped()) {
      stop();
    }
    if (acceptorGroup != null) {
      acceptorGroup.shutdownGracefully(0, 0, SECONDS).syncUninterruptibly();
      acceptorGroup = null;
    }
    if (workerGroup != null) {
      workerGroup.shutdownGracefully(0, 0, SECONDS).syncUninterruptibly();
      workerGroup = null;
    }
    if (onDisposeCallback != null) {
      onDisposeCallback.run();
    }
    if (selectorsScheduler != null) {
      selectorsScheduler.stop();
      selectorsScheduler = null;
    }
  }

  @Override
  public ServerAddress getServerAddress() {
    return serverAddress;
  }

  @Override
  public HttpConstants.Protocol getProtocol() {
    return sslContext == null ? HTTP : HTTPS;
  }

  @Override
  public boolean isStopping() {
    return STOPPING == status;
  }

  @Override
  public boolean isStopped() {
    return STOPPED == status;
  }

  @Override
  public RequestHandlerManager addRequestHandler(Collection<String> methods, String path, RequestHandler requestHandler) {
    return httpListenerRegistry.addRequestHandler(this, requestHandler, PathAndMethodRequestMatcher.builder()
        .methodRequestMatcher(MethodRequestMatcher.builder(methods).build())
        .path(path)
        .build());
  }

  @Override
  public RequestHandlerManager addRequestHandler(String path, RequestHandler requestHandler) {
    return httpListenerRegistry.addRequestHandler(this, requestHandler, PathAndMethodRequestMatcher.builder()
        .methodRequestMatcher(acceptAll())
        .path(path)
        .build());
  }

  @Override
  public WebSocketHandlerManager addWebSocketHandler(WebSocketHandler handler) {
    return webSocketsHandlersRegistry.addWebSocketHandler(handler);
  }

  @Override
  public SseEndpointManager sse(String ssePath,
                                Consumer<SseRequestContext> onRequest,
                                Consumer<SseClient> onClient) {
    return new SseHandlerManagerAdapter(addRequestHandler(ssePath, new SseRequestHandler(onRequest, onClient)));
  }

  public static class Builder {

    private static final int DEFAULT_SELECTORS_COUNT = min(getRuntime().availableProcessors(), 2);

    private ServerAddress serverAddress;
    private HttpListenerRegistry httpListenerRegistry;
    private AcceptedConnectionChannelInitializer clientChannelHandler;
    private WebSocketsHandlersRegistry webSocketsHandlersRegistry = new WebSocketsHandlersRegistry() {};
    private Scheduler selectorsScheduler;
    private int selectorsCount = DEFAULT_SELECTORS_COUNT;
    private Supplier<Long> shutdownTimeout = () -> 5000L;
    private SslContext sslContext;
    private Runnable onDisposeCallback;

    Builder() {}

    public HttpServer build() {
      checkArgument(serverAddress != null, "Server address can't be null");
      checkArgument(httpListenerRegistry != null, "Listener registry can't be null");
      return new NettyHttpServer(serverAddress, httpListenerRegistry, clientChannelHandler, webSocketsHandlersRegistry,
                                 selectorsScheduler, selectorsCount, shutdownTimeout, sslContext, onDisposeCallback);
    }

    public Builder withSslContext(SslContext sslContext) {
      this.sslContext = sslContext;
      return this;
    }

    public Builder withServerAddress(InetSocketAddress socketAddress) {
      this.serverAddress = new DefaultServerAddress(socketAddress.getAddress(), socketAddress.getPort());
      return this;
    }

    public Builder withHttpListenerRegistry(HttpListenerRegistry httpListenerRegistry) {
      this.httpListenerRegistry = httpListenerRegistry;
      return this;
    }

    public Builder withWebSocketsHandlersRegistry(WebSocketsHandlersRegistry webSocketsHandlersRegistry) {
      this.webSocketsHandlersRegistry = webSocketsHandlersRegistry;
      return this;
    }

    public Builder doOnDispose(Runnable onDisposeCallback) {
      this.onDisposeCallback = onDisposeCallback;
      return this;
    }

    public Builder withClientChannelHandler(AcceptedConnectionChannelInitializer clientChannelHandler) {
      this.clientChannelHandler = clientChannelHandler;
      return this;
    }

    public Builder withSelectorsScheduler(Scheduler selectorsScheduler) {
      this.selectorsScheduler = selectorsScheduler;
      return this;
    }

    public Builder withSelectorsCount(int selectorsCount) {
      this.selectorsCount = selectorsCount;
      return this;
    }

    public Builder withShutdownTimeout(Supplier<Long> shutdownTimeout) {
      this.shutdownTimeout = shutdownTimeout;
      return this;
    }
  }
}
