/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.http.api.domain.request.ClientConnection;

import java.net.InetSocketAddress;
import java.security.cert.Certificate;

import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;

import io.netty.handler.ssl.SslHandler;
import org.slf4j.Logger;

public final class NettyClientConnection implements ClientConnection {

  private static final Logger LOGGER = getLogger(NettyClientConnection.class);

  private final InetSocketAddress remoteHostAddress;
  private final SSLSession sslSession;

  public NettyClientConnection(InetSocketAddress remoteHostAddress, SslHandler sslHandler) {
    this.remoteHostAddress = remoteHostAddress;
    this.sslSession = getSslSession(sslHandler);
  }

  private static SSLSession getSslSession(SslHandler sslHandler) {
    if (sslHandler == null) {
      return null;
    }
    SSLEngine engine = sslHandler.engine();
    if (engine == null) {
      return null;
    }
    return engine.getSession();
  }

  @Override
  public InetSocketAddress getRemoteHostAddress() {
    return remoteHostAddress;
  }

  @Override
  public Certificate getClientCertificate() {
    try {
      if (sslSession != null) {
        Certificate[] peerCertificates = sslSession.getPeerCertificates();
        if (peerCertificates.length > 0) {
          return peerCertificates[0];
        }
      }
    } catch (SSLPeerUnverifiedException e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Failure getting peer certificates", e);
      }
    }
    return null;
  }
}
