/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.common.server.sse;

import static org.mule.runtime.http.api.HttpHeaders.Names.CONNECTION;
import static org.mule.runtime.http.api.HttpHeaders.Values.CLOSE;

import static java.util.Optional.ofNullable;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.request.HttpRequestContext;
import org.mule.runtime.http.api.server.async.HttpResponseReadyCallback;
import org.mule.runtime.http.api.sse.server.SseRequestContext;
import org.mule.runtime.http.api.sse.server.SseResponseCustomizer;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;

public class SseRequestContextImpl implements SseRequestContext {

  private final HttpRequestContext requestContext;
  private final HttpResponseReadyCallback responseCallback;
  private String clientId = null;
  private boolean rejectResponseSent;
  private Consumer<SseResponseCustomizer> responseCustomizer;

  public SseRequestContextImpl(HttpRequestContext requestContext, HttpResponseReadyCallback responseCallback) {
    this.requestContext = requestContext;
    this.responseCallback = responseCallback;
    this.rejectResponseSent = false;
    this.responseCustomizer = res -> {
    };
  }

  @Override
  public HttpRequest getRequest() {
    return requestContext.getRequest();
  }

  @Override
  public HttpRequestContext getRequestContext() {
    return requestContext;
  }

  @Override
  public void setClientId(String overrideId) {
    this.clientId = overrideId;
  }

  @Override
  public CompletableFuture<Void> reject(int statusCode, String reasonPhrase) {
    var response = HttpResponse.builder()
        .statusCode(statusCode)
        .reasonPhrase(reasonPhrase)
        .addHeader(CONNECTION, CLOSE)
        .build();
    var future = new CompletableFuture<Void>();
    rejectResponseSent = true;
    responseCallback.responseReady(response, new FutureCompleterCallback(future));
    return future;
  }

  @Override
  public void customizeResponse(Consumer<SseResponseCustomizer> responseCustomizer) {
    this.responseCustomizer = responseCustomizer;
  }

  public Consumer<SseResponseCustomizer> getResponseCustomizer() {
    return responseCustomizer;
  }

  public boolean isResponded() {
    return rejectResponseSent;
  }

  public Optional<String> getClientId() {
    return ofNullable(clientId);
  }
}
