/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.utils;

import static org.slf4j.LoggerFactory.getLogger;

import java.io.IOException;
import java.net.Socket;

import org.slf4j.Logger;

public class TcpTextClient implements AutoCloseable {

  private static final Logger LOGGER = getLogger(TcpTextClient.class);

  private final Socket socket;
  private StringBuilder response = new StringBuilder();

  public TcpTextClient(String host, int port) throws IOException {
    this.socket = new Socket(host, port);
  }

  public void close() throws IOException {
    this.socket.close();
  }

  public void sendString(String message) throws IOException {
    socket.getOutputStream().write(message.getBytes());
    socket.getOutputStream().flush();
  }

  public String receive(int length) throws IOException {
    String resultBeforeRead = getResultIfLengthAvailable(length);
    if (resultBeforeRead != null) {
      return resultBeforeRead;
    }

    byte[] buffer = new byte[1024];
    boolean endOfStreamFound = false;
    while (!endOfStreamFound) {
      int actuallyRead = socket.getInputStream().read(buffer);
      if (actuallyRead == -1) {
        endOfStreamFound = true;
      } else {
        String asString = new String(buffer, 0, actuallyRead);
        LOGGER.debug("Read (with len) '{}'", asString);
        response.append(asString);
        String result = getResultIfLengthAvailable(length);
        if (result != null) {
          return result;
        }
      }
    }
    return response.toString();
  }

  private String getResultIfLengthAvailable(int length) {
    if (response.length() < length) {
      return null;
    }
    String asString = response.toString();
    String result = asString.substring(0, length);
    String remaining = asString.substring(length);
    response = new StringBuilder(remaining);
    return result;
  }

  public String receiveUntil(String delimiter) throws IOException {
    String resultBeforeRead = getResultIfDelimiterFound(delimiter);
    if (resultBeforeRead != null) {
      return resultBeforeRead;
    }

    byte[] buffer = new byte[1024];
    boolean endOfStreamFound = false;
    while (!endOfStreamFound) {
      int actuallyRead = socket.getInputStream().read(buffer);
      if (actuallyRead == -1) {
        endOfStreamFound = true;
      } else {
        String asString = new String(buffer, 0, actuallyRead);
        LOGGER.debug("Receiving (with until) '{}'", asString);
        response.append(asString);
        String result = getResultIfDelimiterFound(delimiter);
        if (result != null) {
          return result;
        }
      }
    }
    String result = response.toString();
    response = new StringBuilder();
    return result;
  }

  private String getResultIfDelimiterFound(String delimiter) {
    String allReceived = response.toString();
    int indexOfDelimiter = allReceived.indexOf(delimiter);
    if (indexOfDelimiter != -1) {
      String result = allReceived.substring(0, indexOfDelimiter);
      String remaining = allReceived.substring(indexOfDelimiter + delimiter.length());
      response = new StringBuilder(remaining);
      return result;
    }
    return null;
  }

  public void shutdownWrite() throws IOException {
    this.socket.shutdownOutput();
  }
}
