/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.client;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.runtime.http.api.HttpConstants.Method.POST;
import static org.mule.tck.util.FileUtils.getClassPathRoot;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Files.newInputStream;

import org.mule.runtime.http.api.client.HttpRequestOptions;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.domain.message.response.HttpResponseBuilder;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.DisplayName;

import io.qameta.allure.Issue;

@DisplayName("Validates request streaming with a large payload")
@Issue("W-14543363")
public class RequestStreamingLargePayloadTestCase extends HttpClientPostStreamingTestCase {

  public RequestStreamingLargePayloadTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  protected InputStream getInputStream() {
    try {
      // this needs to be rewindable
      return newInputStream(getClassPathRoot(RequestStreamingLargePayloadTestCase.class).toPath().resolve("largePayload"));
    } catch (Exception e) {
      throw new AssertionError("Error on loading the large payload file: " + e.toString());
    }
  }

  @Override
  public HttpRequest getRequest() {
    return HttpRequest.builder().method(POST).uri(getUri())
        .entity(new InputStreamHttpEntity(getInputStream())).build();
  }

  @Override
  public HttpRequestOptions getOptions() {
    return HttpRequestOptions.builder().responseTimeout(RESPONSE_TIMEOUT).build();
  }

  @Override
  public HttpResponse doSetUpHttpResponse(HttpRequest request) {
    HttpResponseBuilder response = HttpResponse.builder();
    extractPayload(request);
    response.statusCode(OK.getStatusCode());
    return response.build();
  }

  @Override
  protected String expectedPayload() {
    try {
      return IOUtils.toString(this.getClass().getResourceAsStream("/largePayload"), UTF_8);
    } catch (IOException e) {
      throw new AssertionError("Error on loading the large payload file: " + e.toString());
    }
  }

}
