/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static org.slf4j.LoggerFactory.getLogger;

import java.nio.channels.ClosedChannelException;

import javax.net.ssl.SSLHandshakeException;

import io.netty.handler.codec.DecoderException;
import org.slf4j.Logger;
import reactor.netty.http.client.PrematureCloseException;

public class NettyUtils {

  private static final Logger LOGGER = getLogger(NettyUtils.class);

  /**
   * @param exception The exception to be analyzed.
   * @return {@code true} if the passed exception has to be wrapped as a "Remotely closed" IOException, or {@code false} otherwise
   */
  public static boolean shouldWrapAsRemotelyClosed(Throwable exception) {
    if (isConnectionReset(exception)) {
      return true;
    } else if (exception instanceof ClosedChannelException && hasSuppressedSSLException(exception)) {
      return true;
    } else if (exception instanceof DecoderException) {
      // TODO: This condition does not seem to mean that the connection was always closed remotely,
      // it just works for the moment
      LOGGER.warn("Caught exception. Interpreting it as a Remotely Closed", exception);
      return true;
    } else if (exception instanceof PrematureCloseException) {
      // Thrown by reactor-netty
      return true;
    } else {
      return false;
    }
  }

  private static boolean isConnectionReset(Throwable exception) {
    if (exception == null) {
      return false;
    }

    String message = exception.getMessage();
    if (message == null) {
      return false;
    }

    return message.contains("Connection reset");
  }

  private static boolean hasSuppressedSSLException(Throwable exception) {
    return exception.getSuppressed() != null && exception.getSuppressed().length > 0
        && exception.getSuppressed()[0] instanceof SSLHandshakeException;
  }
}
