/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.server;

import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;

import static java.nio.charset.StandardCharsets.UTF_8;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;

import org.mule.runtime.http.api.server.HttpServer;
import org.mule.service.http.netty.impl.message.content.StringHttpEntity;
import org.mule.service.http.netty.impl.server.AcceptedConnectionChannelInitializer;
import org.mule.service.http.netty.impl.server.NettyHttpServer;
import org.mule.service.http.netty.impl.server.util.HttpListenerRegistry;
import org.mule.service.http.test.netty.tck.ExecutorRule;
import org.mule.service.http.test.netty.utils.NoOpResponseStatusCallback;
import org.mule.service.http.test.netty.utils.ResponseWithoutHeaders;
import org.mule.service.http.test.netty.utils.TcpTextClient;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.DynamicPort;

import java.io.IOException;
import java.net.InetSocketAddress;

import org.apache.commons.io.IOUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Issue;

@Issue("W-17408113")
public class HalfTcpShutdownTestCase extends AbstractMuleTestCase {

  @ClassRule
  public static ExecutorRule executorRule = new ExecutorRule();

  @Rule
  public DynamicPort serverPort = new DynamicPort("serverPort");

  private HttpServer httpServer;

  @Before
  public void setup() throws Exception {
    HttpListenerRegistry listenerRegistry = new HttpListenerRegistry();
    httpServer = NettyHttpServer.builder()
        .withServerAddress(new InetSocketAddress(serverPort.getNumber()))
        .withHttpListenerRegistry(listenerRegistry)
        .withShutdownTimeout(() -> 5000L)
        .withClientChannelHandler(new AcceptedConnectionChannelInitializer(listenerRegistry, true, 30000, 10000L, null, 300,
                                                                           executorRule.getExecutor()))
        .build();
    httpServer.start();
    httpServer.addRequestHandler("/test", (requestContext, responseCallback) -> {
      executorRule.getExecutor().submit(() -> {
        try {
          var asString = IOUtils.toString(requestContext.getRequest().getEntity().getContent(), UTF_8);
          responseCallback.responseReady(new ResponseWithoutHeaders(OK, new StringHttpEntity(asString)),
                                         new NoOpResponseStatusCallback());
        } catch (IOException e) {
          responseCallback.responseReady(new ResponseWithoutHeaders(INTERNAL_SERVER_ERROR, new StringHttpEntity(e.toString())),
                                         new NoOpResponseStatusCallback());
        }
      });
    });
  }

  @After
  public void tearDown() {
    if (!httpServer.isStopped()) {
      httpServer.stop();
    }
    httpServer.dispose();
  }

  @Test
  public void halfShutdownClient() throws IOException, InterruptedException {
    try (TcpTextClient tcpTextClient = new TcpTextClient("localhost", serverPort.getNumber())) {
      // Send a chunked request...
      tcpTextClient.sendString("" +
          "POST /test HTTP/1.1\n" +
          "Host: localhost:" + serverPort.getNumber() + "\n" +
          "Transfer-Encoding: chunked\n" +
          "\n" +
          "2\n" +
          "OK\n" +
          "0\n" +
          "\n");

      // And shutdown output
      tcpTextClient.shutdownWrite();

      // The response should be received normally.
      assertThat(tcpTextClient.receiveUntil("\r\n\r\n"), containsString("content-length: 2"));
      assertThat(tcpTextClient.receive(2), containsString("OK"));
    }
  }
}
